package ru.yandex.disk.search;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Locale;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.string.EnumParser;

public enum DiskField {
    // Common fields list
    // 'id' and 'resource_id' fields skipped because they already
    // handled by KaliRequestContext.prologue(...)
    // 'type' field skipped because it is already
    // handled by DiskDocumentType.writeDocument(...)
    NAME(DiskFieldType.COMMON, true),
    VERSION(DiskFieldType.COMMON, true),
    CTIME(DiskFieldType.COMMON, false),
    MTIME(DiskFieldType.COMMON, false),
    VISIBLE(DiskFieldType.COMMON, false) {
        @Override
        public void writeField(final JsonWriter writer, final String value)
            throws IOException, JsonException
        {
            boolean parsed;
            try {
                parsed = BooleanParser.INSTANCE.apply(value);
            } catch (RuntimeException e) {
                throw new JsonException(
                    "Malformed field \"visible\": " + value);
            }
            writer.key("visible");
            if (parsed) {
                writer.value(1);
            } else {
                writer.value(0);
            }
        }
    },
    // Folder specific fields list
    PARENT_FIDS(DiskFieldType.DIR, true) {
        @Override
        public void writeField(
            final JsonWriter writer,
            final JsonObject value)
            throws IOException, JsonException
        {
            JsonList list = value.asList();
            if (list.size() > 0) {
                writer.key("parent_fid");
                writer.value(list.get(list.size() - 1).asString());
            }
        }
    },
    DIR_KEY(DiskFieldType.DIR, true, KeyField.KEY) {
        @Override
        public void writeField(final JsonWriter writer, final String value)
            throws IOException, JsonException
        {
            DirKey.INSTANCE.writeField(writer, value);
        }
    },
    FOLDER_TYPE(DiskFieldType.DIR, false),
    FID(DiskFieldType.DIR, true),
    SHARED_FOLDER_OWNER(DiskFieldType.DIR, false),

    // File specific fields list
    PARENT_FID(DiskFieldType.FILE, true),
    FILE_KEY(DiskFieldType.FILE, true, KeyField.KEY) {
        @Override
        public void writeField(final JsonWriter writer, final String value)
            throws IOException, JsonException
        {
            FileKey.INSTANCE.writeField(writer, value);
        }
    },
    ETIME(DiskFieldType.FILE, false),
    PHOTOSLICE_TIME(DiskFieldType.FILE, false),
    STID(DiskFieldType.FILE, true),
    PREVIEW_STID(DiskFieldType.FILE, false),
    MIMETYPE(DiskFieldType.FILE, false),
    MEDIA_TYPE(DiskFieldType.FILE, false) {
        @Override
        public void writeField(final JsonWriter writer, final String value)
            throws IOException, JsonException
        {
            DiskMediaType type;
            try {
                type = MEDIA_TYPE_PARSER.apply(value);
            } catch (Exception e) {
                throw new JsonException(
                    "Malformed field \"media_type\": " + value);
            }
            writer.key("mediatype");
            writer.value(type.ordinal());
        }
    },
    MD5(DiskFieldType.FILE, false),
    SIZE(DiskFieldType.FILE, false),
    FOTKI_TAGS(DiskFieldType.FILE, false),
    EXTERNAL_URL(DiskFieldType.FILE, false) {
        @Override
        public void writeField(final JsonWriter writer, final String value)
            throws IOException, JsonException
        {
            try {
                URI uri = new URI(value).parseServerAuthority();
                String host = uri.getHost();
                if (host != null) {
                    writer.key("external_host");
                    writer.value(host);
                }
            } catch (URISyntaxException e) {
                // Save external_url as is
            }
            super.writeField(writer, value);
        }
    },

    PHOTOSLICE_ALBUM_TYPE(DiskFieldType.FILE, false),
    ALBUMS_EXCLUSIONS(DiskFieldType.FILE, false) {
        @Override
        public void writeField(
            final JsonWriter writer,
            final JsonObject value)
            throws IOException, JsonException
        {
            JsonList list = value.asList();
            if (list.size() > 0) {
                StringBuilder sb = new StringBuilder();
                for (JsonObject item: list) {
                    sb.append(item.asString());
                    sb.append('\n');
                }

                if (sb.length() > 0) {
                    sb.setLength(sb.length() - 1);
                }

                writer.key(fieldName());
                writer.value(sb.toString());
            }
        }
    };

    private static final EnumParser<DiskMediaType> MEDIA_TYPE_PARSER =
        new EnumParser<>(DiskMediaType.class);

    private final DiskFieldType type;
    private final boolean mandatory;
    private final String fieldName;

    DiskField(final DiskFieldType type, final boolean mandatory) {
        this.type = type;
        this.mandatory = mandatory;
        fieldName = name().toLowerCase(Locale.ROOT);
    }

    DiskField(
        final DiskFieldType type,
        final boolean mandatory,
        final String fieldName)
    {
        this.type = type;
        this.mandatory = mandatory;
        this.fieldName = fieldName;
    }

    public DiskFieldType type() {
        return type;
    }

    public boolean mandatory() {
        return mandatory;
    }

    public String fieldName() {
        return fieldName;
    }

    public void writeField(final JsonWriter writer, final String value)
        throws IOException, JsonException
    {
        writer.key(fieldName);
        writer.value(value);
    }

    public void writeField(final JsonWriter writer, final JsonObject value)
        throws IOException, JsonException
    {
        writeField(writer, value.asString());
    }

    private interface KeyField {
        String KEY = "key";
        String FOLDER = "folder";

        default void writeField(final JsonWriter writer, final String key)
            throws IOException
        {
            writer.key(KEY);
            writer.value(key);
            int slash = key.indexOf('/', 1);
            if (slash > 1) {
                writer.key("aux_folder");
                writer.value(key.substring(1, slash));
                writePath(writer, key.substring(slash + 1));
            }
        }

        void writePath(JsonWriter writer, String path) throws IOException;
    }

    private enum DirKey implements KeyField {
        INSTANCE;

        @Override
        public void writePath(final JsonWriter writer, final String path)
            throws IOException
        {
            if (!path.isEmpty()) {
                writer.key(FOLDER);
                writer.value(path);
            }
        }
    }

    private enum FileKey implements KeyField {
        INSTANCE;

        @Override
        public void writePath(final JsonWriter writer, final String path)
            throws IOException
        {
            int lastSlash = path.lastIndexOf('/');
            if (lastSlash > 0) {
                writer.key(FOLDER);
                writer.value(path.substring(0, lastSlash));
            }
            int dot = path.lastIndexOf('.');
            if (dot > lastSlash + 1
                && dot + 1 < path.length())
            {
                writer.key("ext");
                writer.value(path.substring(dot + 1));
            }
        }
    }
}

