#include "imageparser_jniwrapper.h"

#include <cv/imageproc/faces/tf_facecropper/face_calculator.h>
#include <cv/imgclassifiers/tf_applicator/models/detection/common/utils.h>
#include <cv/library/configthumb/config.h>
#include <cv/library/imageparserlib/imageparser.h>
#include <cv/library/imagesizes/sizes.h>

#include <extsearch/images/kernel/cbir/imparser_chunk/features_chunk.h>

#include <library/cpp/http/misc/httpreqdata.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/yconf/conf.h>

#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/utility.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>
#include <util/generic/ylimits.h>
#include <util/string/cast.h>
#include <util/string/hex.h>
#include <util/string/type.h>
#include <util/system/compiler.h>
#include <util/system/file.h>

static const TString I2TLayer("prod_v5_enc_i2t_v7_200_img");

DEFINE_SECTION(ImageParserConfig)
    DIRECTIVE(OldImageParserConfig)
    DIRECTIVE(FaceCalculatorConfig)
END_DEFINE_SECTION

DECLARE_CONFIG(TImageParserJniWrapperConfig)
BEGIN_CONFIG(TImageParserJniWrapperConfig)
    BEGIN_TOPSECTION(ImageParserConfig)
    END_SECTION()
END_CONFIG()

TImageParserJniWrapper::TImageParserJniWrapper(const char* config) {
    TImageParserJniWrapperConfig configParser;
    if (!configParser.ParseMemory(config)) {
        ythrow TWithBackTrace<yexception>()
            << "Failed to parse config:\n" << config;
    }
    TYandexConfig::Section* section =
        configParser.GetFirstChild("ImageParserConfig");
    if (!section) {
        ythrow TWithBackTrace<yexception>()
            << "No <ImageParserConfig> found in config:\n" << config;
    }
    const TYandexConfig::Directives& directives = section->GetDirectives();
    TString oldConfigPath;
    if (!directives.GetValue("OldImageParserConfig", oldConfigPath)) {
        ythrow TWithBackTrace<yexception>()
            << "No OldImageParserConfig set in config:\n" << config;
    }
    TString facesConfigPath;
    if (!directives.GetValue("FaceCalculatorConfig", facesConfigPath)) {
        ythrow TWithBackTrace<yexception>()
            << "No FaceCalculatorConfig set in config:\n" << config;
    }

    Config.ExecuteConfig(oldConfigPath.c_str());
    Parser.Init(Config);
    Context.SetDetectFaces(true);
    Context.SetRunNeuralNetClassifier(true);
    Context.SetNetClassifiersLabels(
        "wallpaper,beautiful,bad_quality,good_quality,ok_quality,"
        "bad_quality_v2,good_quality_v2,ok_quality_v2,"
        "pairwise_attraction_0,500px_0,cost_disk_aethetic_0,mobile_porn,"
        "child_porn,gruesome,perversion,binary_porn,multiclass_porn,"
        "prof_porn,porn_with_erotic");
    Context.SetNetFeaturesLayerNames(I2TLayer);

    NConfig::TConfig facesConfig =
        NBBoxDetection::ReadConfig(facesConfigPath);
    FaceCalculator.Reset(new NSimilarFaces::TFaceCalculator(facesConfig));
}

int TImageParserJniWrapper::Parse(const char* uri, const void *data, size_t size, char** out) {
    TServerRequestData rd;
    rd.Parse(uri);
    rd.Scan();
    bool extractFaces = IsTrue(rd.CgiParam.Get("extract-faces"));
    bool oldCv = !IsFalse(rd.CgiParam.Get("old-cv"));
    bool failOnEmpty = !IsFalse(rd.CgiParam.Get("fail-on-empty"));
    Parser.ExtReset();
    int ret = -1;
    if (oldCv) {
        ret = Parser.Parse(size, data, &Context);
    } else if (extractFaces) {
        ret = 0;
    }
    if (ret) {
        TString message =
            "Failed to parse image, code: " + ToString(ret);
        *out = strdup(message.c_str());
        return -2;
    }
    bool empty = true;
    NJsonWriter::TBuf writer(NJsonWriter::HEM_UNSAFE);
    writer.BeginObject();
    if (oldCv) {
        i32 featuresChunkSize;
        const NNeuralNet::TFeaturesChunk* featuresChunk =
            reinterpret_cast<const NNeuralNet::TFeaturesChunk*>(
                Parser.GetResultStore()->GetChunk(
                    NNeuralNet::TFeaturesChunk::CHUNK_ID,
                    featuresChunkSize));
        if (featuresChunk && featuresChunkSize) {
            NNeuralNet::TDLFeatures features(featuresChunk->Data);
            TVector<float> featuresVector;
            features.GetFeatures(I2TLayer, featuresVector);
            size_t size = featuresVector.ysize();
            if (size) {
                empty = false;
                TVector<signed char> hex(size);
                for (size_t i = 0; i < size; ++i) {
                    hex[i] = ClampVal<int>(
                        featuresVector[i] * 128,
                        Min<signed char>(),
                        Max<signed char>());
                }
                writer.WriteKey(TStringBuf("i2t_hex"));
                writer.WriteString(HexEncode(hex.data(), hex.ysize()));
                writer.WriteKey(TStringBuf("i2t_hex_precise"));
                writer.WriteString(
                    HexEncode(
                        featuresVector.data(),
                        featuresVector.ysize() * sizeof featuresVector[0]));
            }
        }
        TPreds predictions = Parser.GetNamedClassesPredictions();
        if (!predictions.empty()) {
            empty = false;
            writer.WriteKey(TStringBuf("classes"));
            writer.BeginObject();
            for (const auto& prediction: predictions) {
                writer.WriteKey(prediction.first);
                writer.WriteFloat(prediction.second);
            }
            writer.EndObject();
        }
        i32 facesChunkSize;
        const PrsAlg::AnnotationFD* facesChunk =
            reinterpret_cast<const PrsAlg::AnnotationFD*>(
                Parser.GetResultStore()->GetChunk(
                    AnnotationFD::CHUNK_ID,
                    facesChunkSize));
        if (facesChunk && facesChunkSize && facesChunk->nFaces) {
            empty = false;
            writer.WriteKey(TStringBuf("faces"));
            writer.BeginList();
            double width = Parser.GetImageConverter()->GetImageWidth();
            double height = Parser.GetImageConverter()->GetImageHeight();
            for (size_t i = 0; i < facesChunk->nFaces; ++i) {
                const PrsAlg::Face& face = facesChunk->faces[i];
                writer.BeginObject();
                writer.WriteKey(TStringBuf("x"));
                writer.WriteDouble(face.x / width);
                writer.WriteKey(TStringBuf("y"));
                writer.WriteDouble(face.y / height);
                writer.WriteKey(TStringBuf("width"));
                writer.WriteDouble(face.width / width);
                writer.WriteKey(TStringBuf("height"));
                writer.WriteDouble(face.height / width);
                writer.EndObject();
            }
            writer.EndList();
        }
    }

    if (extractFaces) {
        //cv::Mat image{cv::imdecode(cv::_InputArray(reinterpret_cast<const ui8*>(data), size), cv::IMREAD_COLOR)};
        TBlob blob = TBlob::NoCopy(data, size);
        cv::Mat image;
        if (!BlobToMat(image, blob)) {
            *out = strdup("Can't parse image blob");
            return -1;
        }
        TVector<NSimilarFaces::TFaceInfo> attrs =
            FaceCalculator->ProcessImage(image);

        double width = image.cols;
        double height = image.rows;
        writer.WriteKey(TStringBuf("face_infos"));
        writer.BeginList();
        if (!attrs.empty()) {
            empty = false;
            for (const auto& face: attrs) {
                writer.BeginObject();
                writer.WriteKey(TStringBuf("x"));
                writer.WriteDouble(face.Detection.Box.Left / width);
                writer.WriteKey(TStringBuf("y"));
                writer.WriteDouble(face.Detection.Box.Top / height);
                writer.WriteKey(TStringBuf("width"));
                writer.WriteDouble(face.Detection.Box.Width / width);
                writer.WriteKey(TStringBuf("height"));
                writer.WriteDouble(face.Detection.Box.Height / height);
                writer.WriteKey(TStringBuf("confidence"));
                writer.WriteFloat(face.Detection.Prob);
                writer.WriteKey(TStringBuf("age"));
                writer.WriteFloat(face.SocdemScores.Age);
                writer.WriteKey(TStringBuf("female_probability"));
                writer.WriteFloat(face.SocdemScores.Female);

                writer.WriteKey(TStringBuf("signature"));
                const TVector<float>& features = face.Features;
                writer.WriteString(
                    HexEncode(
                        features.data(),
                        features.ysize() * sizeof features[0]));

                writer.WriteKey(TStringBuf("image_width"));
                writer.WriteDouble(width);
                writer.WriteKey(TStringBuf("image_height"));
                writer.WriteDouble(height);
                writer.EndObject();
            }
        }

        writer.EndList();
    }

    Parser.ExtReset();
    if (empty && failOnEmpty) {
        *out = nullptr;
        return -2;
    }
    writer.EndObject();
    *out = strdup(writer.Str().c_str());
    return 0;
}

extern "C"
int JniWrapperCreateImageParser(const char* config, void** out) noexcept {
    try {
        *out = new TImageParserJniWrapper(config);
    } catch (...) {
        return NJniWrapper::ProcessJniWrapperException((char**) out);
    }
    return 0;
}

extern "C"
void JniWrapperDestroyImageParser(void* instance) noexcept {
    delete static_cast<TImageParserJniWrapper*>(instance);
}

extern "C"
int JniWrapperParseImage(
    void* instance,
    const char* uri,
    const char* metainfo Y_DECLARE_UNUSED,
    const void* data,
    size_t size,
    char** out) noexcept
{
    auto parser = static_cast<TImageParserJniWrapper*>(instance);
    try {
        return parser->Parse(uri, data, size, out);
    } catch (...) {
        return NJniWrapper::ProcessJniWrapperException(out);
    }
}

