#include "imageparser_jniwrapper.h"

#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

const static TString Config =
    "<ImageParserConfig>\n"
    "\tOldImageParserConfig: imageparser.cfg\n"
    "\tFaceCalculatorConfig: config_mobilenet_cpu.cfg\n"
    "</ImageParserConfig>";

static size_t NextPos(const TString& str, size_t pos) {
    while (true) {
        size_t nextPos = str.find("\":", pos);
        if (nextPos == TString::npos) {
            return TString::npos;
        } else {
            auto c = str[nextPos + 2];
            if (c != '{' && c != '[') {
                return nextPos;
            } else {
                pos = nextPos + 2;
            }
        }
    }
}

static TString ClearValues(const TString& str) {
    size_t pos = NextPos(str, 0);
    TString result{str.substr(0, pos + 2)};
    result.append("\"_placeholder_\"");
    while (true) {
        size_t nextPos = Min(str.find(',', pos), str.find('}', pos));
        pos = NextPos(str, nextPos);
        if (pos == TString::npos) {
            result.append(str.substr(nextPos));
            break;
        } else {
            result.append(str.substr(nextPos, pos + 2 - nextPos));
            result.append("\"_placeholder_\"");
        }
    }
    return result;
}

static TString ParseImage(const TString& filename, const TString& uri) {
    TBlob blob{TBlob::FromFile(filename)};
    TImageParserJniWrapper parser(Config.c_str());
    char* out;
    int ret = parser.Parse(uri.c_str(), blob.Data(), blob.Size(), &out);
    TString result;
    if (out) {
        result = out;
        free(out);
    }
    if (ret) {
        UNIT_FAIL(TString::Join("Parse failed for <", filename, ">, ret = ", ToString(ret), ", reason: ", result));
    }
    return ClearValues(result);
}

static TString LoadCanonicalDataFor(const TString& filename) {
    TString canonical{TString::Join(SRC_("resources"), "/", filename, ".json")};
    TBlob blob{TBlob::FromFile(canonical)};
    return TString{blob.AsCharPtr(), blob.Size()};
}

static void TestFile(const TString& filename, const TString& uri) {
    UNIT_ASSERT_STRINGS_EQUAL(LoadCanonicalDataFor(filename), ParseImage(filename, uri));
}

Y_UNIT_TEST_SUITE(TestImageParer) {
    Y_UNIT_TEST(PreviewTest) {
        //TestFile("2a0000016e74d798a9a3a35d22f42e47ffab.webp", "/?extract-faces=true&old-cv=false&fail-on-empty=false");
        TestFile("big_preview.jpg", "/?extract-faces=true&old-cv=false");
    }
    Y_UNIT_TEST(AvatarTest) {
        TestFile("avatar-old.jpg", "/?extract-faces=true");
    }
    Y_UNIT_TEST(AvatarOnlyOldTest) {
        UNIT_ASSERT_STRINGS_EQUAL(LoadCanonicalDataFor("avatar-only-old.jpg"), ParseImage("avatar-old.jpg", "/"));
    }
    Y_UNIT_TEST(AvatarOnlyFacesTest) {
        UNIT_ASSERT_STRINGS_EQUAL(LoadCanonicalDataFor("avatar-only-face.jpg"), ParseImage("avatar-old.jpg", "/?old-cv=false&extract-faces=true"));
    }
    Y_UNIT_TEST(AvatarMediumTest) {
        TestFile("avatar-medium.jpg", "/?extract-faces=true");
    }
    Y_UNIT_TEST(ScreenshotTest) {
        TestFile("screenshot.png", "/?extract-faces=true");
    }
    Y_UNIT_TEST(CatTest) {
        TestFile("sarah.jpg", "/?extract-faces=true");
    }
    Y_UNIT_TEST(JniWrapperTest) {
        void *instance;
        UNIT_ASSERT_VALUES_EQUAL(JniWrapperCreateImageParser(Config.c_str(), &instance), 0);
        TBlob blob{TBlob::FromFile("spb.webp")};
        char* out;
        UNIT_ASSERT_VALUES_EQUAL(
            JniWrapperParseImage(instance, "/?extract-faces=true", nullptr, blob.Data(), blob.Size(), &out),
            0);
        TString result{ClearValues((const char*) out)};
        UNIT_ASSERT_STRINGS_EQUAL(LoadCanonicalDataFor("spb.webp"), result);
        JniWrapperFree(out);
        JniWrapperDestroyImageParser(instance);
    }
}

