package ru.yandex.search.disk.kali;

import java.io.File;

import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.URIConfig;
import ru.yandex.http.config.URIConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.search.disk.kali.config.KaliFaceConfig;
import ru.yandex.search.disk.kali.config.KaliFaceConfigBuilder;

public abstract class AbstractKaliConfigBuilder
    <T extends AbstractKaliConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements KaliConfig
{
    private URIConfigBuilder djfsConfig;
    private HttpHostConfigBuilder tikaiteConfig;
    private int bodyTextLimit;
    private HttpHostConfigBuilder searcherConfig;
    private HttpHostConfigBuilder indexerConfig;
    private HttpHostConfigBuilder ocrProxyConfig;
    private HttpTargetConfigBuilder callbacksConfig;
    private KaliFaceConfigBuilder faceConfig;
    private String callbacksQueue;
    private Tvm2ClientConfigBuilder tvm2ClientConfig;
    private String djfsTvmClientId;
    private String tikaiteTvmClientId;
    private String unistorageTvmClientId;
    private File peronsalUserLimits;

    protected AbstractKaliConfigBuilder(final KaliConfig config) {
        super(config);
        djfsConfig(config.djfsConfig());
        tikaiteConfig(config.tikaiteConfig());
        bodyTextLimit(config.bodyTextLimit());
        searcherConfig(config.searcherConfig());
        indexerConfig(config.indexerConfig());
        ocrProxyConfig(config.ocrProxyConfig());
        callbacksConfig(config.callbacksConfig());
        callbacksQueue(config.callbacksQueue());
        tvm2ClientConfig(config.tvm2ClientConfig());
        djfsTvmClientId(config.djfsTvmClientId());
        tikaiteTvmClientId(config.tikaiteTvmClientId());
        unistorageTvmClientId(config.unistorageTvmClientId());
        faceConfig(config.faceConfig());
        peronsalUserLimits(config.peronsalUserLimits());
    }

    protected AbstractKaliConfigBuilder(
        final IniConfig config,
        final KaliConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        djfsConfig = new URIConfigBuilder(
            config.section("djfs"),
            defaults.djfsConfig());
        IniConfig tikaiteSection = config.section("tikaite");
        tikaiteConfig = new HttpHostConfigBuilder(
            tikaiteSection,
            defaults.tikaiteConfig());
        bodyTextLimit =
            BODY_TEXT_LIMIT.extract(tikaiteSection, defaults.bodyTextLimit());
        searcherConfig = new HttpHostConfigBuilder(
            config.section("searcher"),
            defaults.searcherConfig());
        indexerConfig = new HttpHostConfigBuilder(
            config.section("indexer"),
            defaults.indexerConfig());
        ocrProxyConfig = new HttpHostConfigBuilder(
            config.section("ocr-proxy"),
            defaults.ocrProxyConfig());
        IniConfig callbacksSection = config.section("callbacks");
        callbacksConfig = new HttpTargetConfigBuilder(
            callbacksSection,
            defaults.callbacksConfig());
        callbacksQueue = CALLBACKS_QUEUE.extract(
            callbacksSection,
            defaults.callbacksQueue());
        tvm2ClientConfig = new Tvm2ClientConfigBuilder(
            config.section("tvm2"),
            defaults.tvm2ClientConfig());

        IniConfig faceSection = config.sectionOrNull("face");
        if (faceSection != null) {
            faceConfig = new KaliFaceConfigBuilder(faceSection, defaults.faceConfig());
        } else {
            faceConfig = new KaliFaceConfigBuilder(defaults.faceConfig());
        }

        djfsTvmClientId =
            DJFS_TVM_CLIENT_ID.extract(config, defaults.djfsTvmClientId());
        tikaiteTvmClientId = TIKAITE_TVM_CLIENT_ID.extract(
            config,
            defaults.tikaiteTvmClientId());
        unistorageTvmClientId = UNISTORAGE_TVM_CLIENT_ID.extract(
            config,
            defaults.unistorageTvmClientId());

        peronsalUserLimits =
            config.getInputFile("personal-user-limits-file", defaults.peronsalUserLimits());
    }

    @Override
    public URIConfigBuilder djfsConfig() {
        return djfsConfig;
    }

    public T djfsConfig(final URIConfig djfsConfig) {
        this.djfsConfig = new URIConfigBuilder(djfsConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder tikaiteConfig() {
        return tikaiteConfig;
    }

    public T tikaiteConfig(final HttpHostConfig tikaiteConfig) {
        this.tikaiteConfig = new HttpHostConfigBuilder(tikaiteConfig);
        return self();
    }

    @Override
    public int bodyTextLimit() {
        return bodyTextLimit;
    }

    public T bodyTextLimit(final int bodyTextLimit) {
        this.bodyTextLimit = bodyTextLimit;
        return self();
    }

    @Override
    public HttpHostConfigBuilder searcherConfig() {
        return searcherConfig;
    }

    public T searcherConfig(final HttpHostConfig searcherConfig) {
        this.searcherConfig = new HttpHostConfigBuilder(searcherConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder indexerConfig() {
        return indexerConfig;
    }

    public T indexerConfig(final HttpHostConfig indexerConfig) {
        this.indexerConfig = new HttpHostConfigBuilder(indexerConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder ocrProxyConfig() {
        return ocrProxyConfig;
    }

    public T ocrProxyConfig(final HttpHostConfig ocrProxyConfig) {
        this.ocrProxyConfig = new HttpHostConfigBuilder(ocrProxyConfig);
        return self();
    }

    @Override
    public HttpTargetConfigBuilder callbacksConfig() {
        return callbacksConfig;
    }

    public T callbacksConfig(final HttpTargetConfig callbacksConfig) {
        this.callbacksConfig = new HttpTargetConfigBuilder(callbacksConfig);
        return self();
    }

    @Override
    public String callbacksQueue() {
        return callbacksQueue;
    }

    public T callbacksQueue(final String callbacksQueue) {
        this.callbacksQueue = callbacksQueue;
        return self();
    }

    @Override
    public Tvm2ClientConfigBuilder tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public T tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig) {
        this.tvm2ClientConfig = new Tvm2ClientConfigBuilder(tvm2ClientConfig);
        return self();
    }

    @Override
    public String djfsTvmClientId() {
        return djfsTvmClientId;
    }

    public T djfsTvmClientId(final String djfsTvmClientId) {
        this.djfsTvmClientId = djfsTvmClientId;
        return self();
    }

    @Override
    public String tikaiteTvmClientId() {
        return tikaiteTvmClientId;
    }

    public T tikaiteTvmClientId(final String tikaiteTvmClientId) {
        this.tikaiteTvmClientId = tikaiteTvmClientId;
        return self();
    }

    @Override
    public String unistorageTvmClientId() {
        return unistorageTvmClientId;
    }

    public T unistorageTvmClientId(final String unistorageTvmClientId) {
        this.unistorageTvmClientId = unistorageTvmClientId;
        return self();
    }

    @Override
    public KaliFaceConfigBuilder faceConfig() {
        return faceConfig;
    }

    public T faceConfig(final KaliFaceConfig faceConfig) {
        this.faceConfig = new KaliFaceConfigBuilder(faceConfig);
        return self();
    }

    @Override
    public File peronsalUserLimits() {
        return peronsalUserLimits;
    }

    public T peronsalUserLimits(final File peronsalUserLimits) {
        this.peronsalUserLimits = peronsalUserLimits;
        return self();
    }
}

