package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.util.EnumSet;
import java.util.Locale;
import java.util.Set;

import org.apache.http.HttpException;

import ru.yandex.concurrent.AsyncLockHolder;
import ru.yandex.disk.search.DiskField;
import ru.yandex.disk.search.DiskFieldType;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.ValuesStorage;

public enum DiskDocumentType {
    DIR(DiskFieldType.DIR) {
        @Override
        public KaliTask createIndexationTask(
            final AsyncLockHolder lock,
            final KaliDocumentContext doc)
            throws HttpException, IOException, JsonException
        {
            return new UpdateDirTask(lock, doc);
        }

        @Override
        public <E extends Exception> String stid(final ValuesStorage<E> doc) {
            return null;
        }
    },
    FILE(DiskFieldType.FILE) {
        @Override
        public KaliTask createIndexationTask(
            final AsyncLockHolder lock,
            final KaliDocumentContext doc)
            throws HttpException
        {
            return new UpdateFileTask(lock, doc);
        }

        @Override
        public <E extends Exception> String stid(final ValuesStorage<E> doc)
            throws E
        {
            return doc.get(
                DiskField.STID.fieldName(),
                NonEmptyValidator.INSTANCE);
        }
    };

    private final String type = name().toLowerCase(Locale.ROOT);
    @SuppressWarnings("ImmutableEnumChecker")
    private final DiskField[] fields;

    DiskDocumentType(final DiskFieldType type) {
        Set<DiskField> fields = EnumSet.noneOf(DiskField.class);
        for (DiskField field: DiskField.values()) {
            DiskFieldType fieldType = field.type();
            if (fieldType == DiskFieldType.COMMON || fieldType == type) {
                fields.add(field);
            }
        }
        this.fields = fields.toArray(new DiskField[fields.size()]);
    }

    public abstract KaliTask createIndexationTask(
        final AsyncLockHolder lock,
        final KaliDocumentContext doc)
        throws HttpException, IOException, JsonException;

    public abstract <E extends Exception> String stid(
        final ValuesStorage<E> doc)
        throws E;

    public void writeDocument(
        final JsonMap doc,
        final JsonWriter writer,
        final DocumentErrorHandler errorHandler)
        throws IOException
    {
        // Type
        writer.key("type");
        writer.value(type);

        // Other fields
        for (DiskField field: fields) {
            String name = field.fieldName();
            try {
                if (field.mandatory()) {
                    JsonObject valueObj = doc.get(name);
                    if (valueObj == JsonNull.INSTANCE) {
                        throw new JsonException(
                            "Attribute " + name
                                + " is not set, but it is mandatory");
                    }

                    field.writeField(writer, valueObj);
                } else {
                    JsonObject valueObj = doc.get(name);

                    if (valueObj != JsonNull.INSTANCE) {
                        field.writeField(writer, valueObj);
                    }
                }
            } catch (JsonException e) {
                errorHandler.handle(name, doc, e);
            }
        }
    }

    @Override
    public String toString() {
        return type;
    }
}

