package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.disk.search.DiskField;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.PassPayloadThroughFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.DiskFaceActionType;
import ru.yandex.search.disk.DiskFaceActionTypeParser;
import ru.yandex.search.disk.DiskIndexAction;
import ru.yandex.util.string.HexStrings;

public class FaceSubprocessor extends FilterFutureCallback<List<Object>> {
    private final String baseFaceUri;
    private final KaliRequestContext context;

    public FaceSubprocessor(
        final KaliRequestContext context,
        final FutureCallback<? super List<Object>> callback)
        throws BadRequestException
    {
        super(callback);

        Kali kali = context.kali();

        QueryConstructor qc =
            new QueryConstructor(
                kali.config().faceConfig().callbackBaseUri());
        qc.append("service", kali.config().faceConfig().faceQueue());
        qc.append("prefix", context.prefix());
        qc.append("ref_queue_id", context.cgiZooQueueId());
        qc.append("ref_queue", context.zooQueue());
        qc.copy(context.session().params(), "action");
        qc.copyIfPresent(context.session().params(), "id");
        baseFaceUri = qc.toString();
        this.context = context;
    }

    @Override
    public void completed(final List<Object> objects) {
        try {
            switch (context.actionType()) {
                case RM:
                case TRASH_DROP_ALL:
                case TRASH_DROP_ELEMENT:
                case CLEAN_TRASH:
                    handleRemove(objects);
                    break;
                default:
                    callback.completed(objects);
                    break;
            }
        } catch (IOException | BadRequestException e) {
            failed(e);
            return;
        }
    }

    public URI processUpdate(
        final Image image,
        final KaliDocumentContext doc)
        throws BadRequestException, URISyntaxException
    {
        // apply filter
        String key = doc.doc().getOrNull(DiskField.FILE_KEY.fieldName());
        String etime = doc.doc().getOrNull(DiskField.ETIME.fieldName());

        String albumType = doc.doc().getString(DiskField.PHOTOSLICE_ALBUM_TYPE.fieldName(), null);
        if (key == null || "screenshots".equalsIgnoreCase(albumType)) {
            context.session().logger().info("Screenshots or null key, skipping face extraction");
            return null;
        }


        if (!key.startsWith("/photounlim/")
            && (etime == null || etime.isEmpty())
            && !"image/heif".equalsIgnoreCase(doc.mimetype())
            && !"image/heic".equalsIgnoreCase(doc.mimetype()))
        {
            context.session().logger().info("Not photounlim and etime null, skipping face extraction");
            return null;
        }

        return buildUpdateUri(image);
    }

    private URI buildUpdateUri(final Image image) throws BadRequestException, URISyntaxException {
        QueryConstructor qc =
            new QueryConstructor(context.kali().config().faceConfig().host().toURI() + baseFaceUri);
        qc.append("resource_id", image.resourceId());
        qc.append("stid", image.stid());
        if (image.previewStid() != null) {
            qc.append("preview_stid", image.previewStid());
        }
        qc.append("width", image.width());
        qc.append("height", image.height());

        return new URI(qc.toString());
    }

    private void handleRemove(
        final List<Object> response)
        throws BadRequestException, IOException
    {
        String queueNamePrefix = HexStrings.LOWER.toString(
            context.zooQueue().getBytes(StandardCharsets.UTF_8));
        StringBuilder hash = new StringBuilder(queueNamePrefix);
        hash.append('f');
        String zooQueueIdHex = Long.toHexString(context.cgiZooQueueId());
        for (int i = zooQueueIdHex.length(); i < KaliRequestContext.QUEUE_ID_PADDING; ++i) {
            hash.append('0');
        }
        hash.append(zooQueueIdHex);
        hash.append('f');

        context.session().logger().info("Response before face: " + String.valueOf(response));

        boolean trashAppend = context.actionType() == DiskIndexAction.TRASH_APPEND
            || context.actionType() == DiskIndexAction.RM;

        DecodableByteArrayOutputStream faceData = null;
        JsonWriterBase writer = null;
        for (Object itemObj: response) {
            if (trashAppend && itemObj instanceof KaliDocumentContext) {
                if (faceData == null) {
                    faceData = new DecodableByteArrayOutputStream();
                    writer = JsonType.NORMAL.create(faceData);
                    writer.startArray();
                }
                KaliDocumentContext doc = (KaliDocumentContext) itemObj;
                writer.startObject();
                writer.key("id");
                writer.value(doc.id());
                writer.key("resource_id");
                writer.value(doc.resourceId());
                writer.key("uid");
                writer.value(context.prefix());
                writer.key("stid");
                writer.value(doc.stid());
                writer.key("preview_stid");
                writer.value(doc.previewStid());
                writer.endObject();
            } else if (trashAppend && itemObj instanceof KaliRequestDoc) {
                KaliRequestDoc doc = (KaliRequestDoc) itemObj;
                if (faceData == null) {
                    faceData = new DecodableByteArrayOutputStream();
                    writer = JsonType.NORMAL.create(faceData);
                    writer.startArray();
                }

                writer.startObject();
                writer.key("id");
                writer.value(doc.id());
                writer.key("resource_id");
                writer.value(doc.resourceId());
                writer.key("uid");
                writer.value(context.prefix());
                writer.endObject();
            }
            if (!(itemObj instanceof Image)) {
                continue;
            }

            Image image = (Image) itemObj;

            if (faceData == null) {
                faceData = new DecodableByteArrayOutputStream();
                writer = JsonType.NORMAL.create(faceData);
                writer.startArray();
            }

            image.writeFaceJson(writer);
        }

        ProxySession session = context.session();
        Kali kali = context.kali();
        QueryConstructor qc =
            new QueryConstructor(baseFaceUri);
        qc.append("service", kali.config().faceConfig().faceQueue());
        qc.append("prefix", context.prefix());
        qc.append("ref_queue_id", context.cgiZooQueueId());
        qc.append("ref_queue", context.zooQueue());
        qc.copy(session.params(), "action");
        qc.copyIfPresent(session.params(), "id");
        session.logger().info("Face callback " + qc.toString());

        if (faceData != null) {
            writer.endArray();
            writer.close();

            BasicAsyncRequestProducerGenerator generator
                = new BasicAsyncRequestProducerGenerator(
                qc.toString(),
                faceData.toByteArray(),
                ContentType.APPLICATION_JSON.withCharset(
                    StandardCharsets.UTF_8));

            generator.addHeader(
                HeaderUtils.createHeader(
                    YandexHeaders.ZOO_HASH,
                    new String(hash)));
            generator.addHeader(
                HeaderUtils.createHeader(
                    YandexHeaders.SERVICE,
                    kali.config().faceConfig().faceQueue()));
            generator.addHeader(
                HeaderUtils.createHeader(
                    YandexHeaders.ZOO_SHARD_ID,
                    Long.toString(context.zooShardId())));
            generator.addHeader(KaliRequestContext.CHECK_DUPLICATE);

            AsyncClient faceClient =
                kali.faceClient().adjust(session.context());

            session.logger().info("Launching to " + kali.config().faceConfig().host() + " " + qc.toString());
            faceClient.execute(
                kali.config().faceConfig().host(),
                generator,
                EmptyAsyncConsumerFactory.OK,
                session.listener().createContextGeneratorFor(faceClient),
                new UpstreamStaterFutureCallback<>(
                    new PassPayloadThroughFutureCallback<>(response, callback),
                    kali.callbacksStater()));
        } else {
            session.logger().info("Face callback null no data");
            callback.completed(response);
        }
    }

    public static FaceSubprocessor createOrNull(
        final KaliRequestContext context,
        final FutureCallback<? super List<Object>> callback)
        throws BadRequestException
    {
        DiskFaceActionType faceActionType = DiskFaceActionTypeParser.actionToFace(context.actionType());

        boolean clusterizeFace = context.session().params().getBoolean("clusterize_face", false);

        if ((faceActionType != DiskFaceActionType.IGNORE)
            && (clusterizeFace || context.kali().config().faceConfig().enabled()))
        {
            return new FaceSubprocessor(context, callback);
        }

        context.session().logger().info("Face update disabled " + faceActionType);
        return null;
    }
}
