package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.AsyncLock;
import ru.yandex.concurrent.AsyncLockHolder;
import ru.yandex.concurrent.LockStorage;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.kali.config.PersonalIndexationLimits;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.LockStorageSizeStater;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.NonEmptyStater;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.util.string.StringUtils;

public class Kali
    extends HttpProxy<ImmutableKaliConfig>
    implements HttpAsyncRequestHandler<JsonObject>
{
    public static final int MAX_CLEANUP_REMOVE_TASKS = 10000;

    private static final Long ZERO = 0L;
    private static final Long ONE = 1L;

    private final LockStorage<String, AsyncLock> lockStorage =
        new LockStorage<>();
    private final AsyncClient djfsClient;
    private final AsyncClient tikaiteClient;
    private final AsyncClient searcherClient;
    private final AsyncClient indexerClient;
    private final ContentType indexerContentType;
    private final AsyncClient ocrProxyClient;
    private final AsyncClient callbacksClient;
    private final AsyncClient faceClient;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;
    private final TimeFrameQueue<Long> tikaiteRequests;
    private final TimeFrameQueue<Object> documentSerializationFailures;
    private final TimeFrameQueue<Long> images;
    private final TimeFrameQueue<Long> removals;
    private final TimeFrameQueue<Long> diskNotFounds;
    private final TimeFrameQueue<DiskDocumentType> documentTypes;
    private final TimeFrameQueue<Integer> lockQueueSize;
    private final TimeFrameQueue<Long> lags;
    private final TimeFrameQueue<Long> skippedRequests;
    private final TimeFrameQueue<Object> malformedActionOrder;
    private final UpstreamStater tikaiteStater;
    private final UpstreamStater djfsStater;
    private final UpstreamStater callbacksStater;
    private final PersonalIndexationLimits personalIndexationLimits;

    public Kali(final ImmutableKaliConfig config)
        throws GeneralSecurityException,
            HttpException,
            IOException,
            JsonException,
            URISyntaxException
    {
        super(config);
        djfsClient = client("DJFS", config.djfsConfig());
        tikaiteClient = client("Tikaite", config.tikaiteConfig());
        searcherClient = client("Searcher", config.searcherConfig());
        indexerClient = client("Indexer", config.indexerConfig());
        indexerContentType = ContentType.APPLICATION_JSON.withCharset(
            config.indexerConfig().requestCharset());
        ocrProxyClient = client("OcrProxy", config.ocrProxyConfig());
        callbacksClient = client("Callbacks", config.callbacksConfig());
        faceClient = client("FaceCallback", config.faceConfig());
        tvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("tvm2"),
            serviceContextRenewalTask,
            config.tvm2ClientConfig());

        tikaiteRequests = new TimeFrameQueue<>(config.metricsTimeFrame());
        documentSerializationFailures =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        images = new TimeFrameQueue<>(config.metricsTimeFrame());
        removals = new TimeFrameQueue<>(config.metricsTimeFrame());
        diskNotFounds = new TimeFrameQueue<>(config.metricsTimeFrame());
        documentTypes = new TimeFrameQueue<>(config.metricsTimeFrame());
        lockQueueSize = new TimeFrameQueue<>(config.metricsTimeFrame());
        lags = new TimeFrameQueue<>(config.metricsTimeFrame());
        skippedRequests = new TimeFrameQueue<>(config.metricsTimeFrame());
        malformedActionOrder = new TimeFrameQueue<>(config.metricsTimeFrame());

        tikaiteStater =
            new UpstreamStater(config.metricsTimeFrame(), "tikaite");
        djfsStater = new UpstreamStater(config.metricsTimeFrame(), "djfs");
        callbacksStater =
            new UpstreamStater(config.metricsTimeFrame(), "callbacks");

        personalIndexationLimits = new PersonalIndexationLimits(config);
        register(
            new Pattern<>("/locks-status", false),
            new LocksStatusHandler(lockStorage));
        register(new Pattern<>("", true), this);

        SearchBackendProxyWithCallbacksHandler luceneProxyHandler =
            new SearchBackendProxyWithCallbacksHandler(this);

        register(new Pattern<>("/modify", false), luceneProxyHandler);
        register(new Pattern<>("/update", false), luceneProxyHandler);
        register(new Pattern<>("/delete", false), luceneProxyHandler);

        registerStater(tikaiteStater);
        registerStater(
            new PassiveStaterAdapter<>(
                tikaiteRequests,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "tikaite-failed-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "tikaite-total-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                documentSerializationFailures,
                new NamedStatsAggregatorFactory<>(
                    "document-serialization-failures_ammm",
                    CountAggregatorFactory.INSTANCE)));
        registerStater(djfsStater);
        registerStater(callbacksStater);
        registerStater(
            new PassiveStaterAdapter<>(
                images,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "image-modification-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "all-modification-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                removals,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "file-removal-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "file-processing-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                diskNotFounds,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "disk-resource-not-found_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "disk-resource-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                documentTypes,
                new EnumStaterFactory<>(
                    type -> "disk-document-type-" + type + "_ammm",
                    DiskDocumentType.values())));
        registerStater(
            new PassiveStaterAdapter<>(
                lockQueueSize,
                new NamedStatsAggregatorFactory<>(
                    "lock-queue-size_axxx",
                    new MaxAggregatorFactory(0L))));
        registerStater(
            new LockStorageSizeStater("lock-storage-size_axxx", lockStorage));
        registerStater(
            new NonEmptyStater("active-indexers_ammv", lags));
        registerStater(
            new PassiveStaterAdapter<>(
                lags,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "indexation-lag_axxx",
                        new MaxAggregatorFactory(0L)),
                    new NamedStatsAggregatorFactory<>(
                        "indexations_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                skippedRequests,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "version-skipped-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "version-checked-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                malformedActionOrder,
                new NamedStatsAggregatorFactory<>(
                    "malformed-action-order_ammm",
                    CountAggregatorFactory.INSTANCE)));
    }

    @Override
    public void start() throws IOException {
        tvm2RenewalTask.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        tvm2RenewalTask.cancel();
        super.close();
    }

    public AsyncClient faceClient() {
        return faceClient;
    }

    public AsyncClient tikaiteClient() {
        return tikaiteClient;
    }

    public AsyncClient searcherClient() {
        return searcherClient;
    }

    public AsyncClient indexerClient() {
        return indexerClient;
    }

    public ContentType indexerContentType() {
        return indexerContentType;
    }

    public AsyncClient ocrProxyClient() {
        return ocrProxyClient;
    }

    public AsyncClient callbacksClient() {
        return callbacksClient;
    }

    public String djfsTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.djfsTvmClientId());
    }

    public String tikaiteTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.tikaiteTvmClientId());
    }

    public String unistorageTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.unistorageTvmClientId());
    }

    public void tikaiteCompleted() {
        tikaiteRequests.accept(ZERO);
    }

    public void tikaiteFailed() {
        tikaiteRequests.accept(ONE);
    }

    public void documentSerializationFailed() {
        documentSerializationFailures.accept(ZERO);
    }

    public void imageModification() {
        images.accept(ONE);
    }

    public void fileModification() {
        images.accept(ZERO);
    }

    public void accountRemoval() {
        removals.accept(ONE);
    }

    public void accountUpdate() {
        removals.accept(ZERO);
    }

    public void diskDocumentNotFound() {
        diskNotFounds.accept(ONE);
    }

    public void diskDocumentFound() {
        diskNotFounds.accept(ZERO);
    }

    public void documentType(final DiskDocumentType type) {
        documentTypes.accept(type);
    }

    public void lockQueueSize(final int lockQueueSize) {
        this.lockQueueSize.accept(lockQueueSize);
    }

    public void requestSkipped() {
        skippedRequests.accept(ONE);
    }

    public void requestAccepted() {
        skippedRequests.accept(ZERO);
    }

    public void malformedActionOrder() {
        malformedActionOrder.accept(ZERO);
    }

    public UpstreamStater tikaiteStater() {
        return tikaiteStater;
    }

    public UpstreamStater callbacksStater() {
        return callbacksStater;
    }

    public LockStorage<String, AsyncLock> lockStorage() {
        return lockStorage;
    }

    public void lag(final long lag) {
        lags.accept(lag);
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(this, exchange, context);
        handle(new KaliRequestContext(this, session), payload);
    }

    private void handle(
        final KaliRequestContext context,
        final JsonObject payload)
        throws HttpException
    {
        KaliActionType actionType = context.kaliActionType();
        List<KaliRequestDoc> docs;
        if (payload == JsonNull.INSTANCE) {
            docs = Collections.singletonList(
                new KaliRequestDoc(actionType, context.session().params()));
        } else {
            try {
                JsonList jsonDocs = payload.get("docs").asList();
                docs = new ArrayList<>(jsonDocs.size());
                for (JsonObject doc: jsonDocs) {
                    docs.add(new KaliRequestDoc(actionType, doc.asMap()));
                }
                if (docs.isEmpty()) {
                    throw new BadRequestException("At least one doc required");
                }
            } catch (JsonException e) {
                throw new BadRequestException(
                    "Failed to parse payload:\n"
                    + JsonType.HUMAN_READABLE.toString(payload),
                    e);
            }
        }
        switch (context.cleanupType()) {
            case OUTER:
                handleOuterCleanup(context, docs);
                break;
            case INNER:
                handleInnerCleanup(context, docs);
                break;
            default:
                handle(context, docs);
                break;
        }
    }

    private void handle(
        final KaliRequestContext context,
        final List<KaliRequestDoc> docs)
        throws HttpException
    {
        // Depending on action type, retrieve some additional data and
        // create List<? extends KaliRequestDoc> each with its own action type:
        //    - for removal, use existing docs
        //    - for indexation, use data retrieved from DJFS
        //    - for search update, use data retrieved from DJFS for indexing
        //      and use unwanted docs found in index for removal
        switch (context.kaliActionType()) {
            case REMOVE:
                handleRemoval(context, docs);
                break;
            default:
                handleUpdate(context, docs, Collections.emptyList());
                break;
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void handleOuterCleanup(
        final KaliRequestContext context,
        final List<KaliRequestDoc> docs)
        throws HttpException
    {
        context.indexerClient().execute(
            config.indexerConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                "/delete?cleanup-elders&text=type:(file+dir)+AND+NOT"
                + "+resource_id:*&prefix=" + context.prefix()),
            EmptyAsyncConsumerFactory.INSTANCE,
            context.session().listener().createContextGeneratorFor(
                context.indexerClient()),
            new EldersCleanupCallback(context, docs));
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void handleInnerCleanup(
        final KaliRequestContext context,
        final List<KaliRequestDoc> docs)
        throws HttpException
    {
        QueryConstructor query = new QueryConstructor(
            "/search-kali?json-type=dollar&get=id,version,resource_id");
        query.append("IO_PRIO", context.ioPrio());
        query.append("prefix", context.prefix());
        query.append("length", MAX_CLEANUP_REMOVE_TASKS);
        query.append(
            "text",
            "resource_id:[" + docs.get(0).resourceId()
            + " TO " + docs.get(docs.size() - 1).resourceId()
            + ']');
        query.append(
            "postfilter",
            "version <= " + context.session().params().getLong("version"));
        context.searcherClient().execute(
            config.searcherConfig().host(),
            new BasicAsyncRequestProducerGenerator(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener()
                .createContextGeneratorFor(context.searcherClient()),
            new KaliCleanupCallback(context, docs));
    }

    public KaliTask createDeleteDocTask(
        final KaliRequestContext context,
        final KaliRequestDoc doc,
        final GenericAutoCloseableChain<RuntimeException> chain)
        throws HttpException
    {
        AsyncLockHolder lock = new AsyncLockHolder(
            lockStorage,
            StringUtils.concat(
                Long.toString(context.prefix()),
                '#',
                doc.resourceId()));
        chain.add(lock);
        return new DeleteDocTask(lock, context, doc);
    }

    private void handleRemoval(
        final KaliRequestContext context,
        final List<KaliRequestDoc> docs)
        throws HttpException
    {
        try (GenericAutoCloseableHolder<
                RuntimeException,
                GenericAutoCloseableChain<RuntimeException>> chain =
                new GenericAutoCloseableHolder<>(
                    new GenericAutoCloseableChain<>()))
        {
            List<KaliTask> tasks = new ArrayList<>(docs.size());
            for (KaliRequestDoc doc: docs) {
                accountRemoval();
                tasks.add(createDeleteDocTask(context, doc, chain.get()));
            }
            context.callback().done();
            for (KaliTask task: tasks) {
                task.execute();
            }
            chain.release();
        } catch (RuntimeException e) {
            context.callback().failed(
                new ServiceUnavailableException(
                    "Something went terribly wrong",
                    e));
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void handleUpdate(
        final KaliRequestContext context,
        final List<KaliRequestDoc> docsForUpdate,
        final List<KaliRequestDoc> docsForRemoval)
        throws HttpException
    {
        ResourceCallback callback =
            new ResourceCallback(context, docsForRemoval);
        if (docsForUpdate.isEmpty()) {
            JsonMap fakeResponse = new JsonMap(BasicContainerFactory.INSTANCE);
            fakeResponse.put("items", JsonList.EMPTY);
            callback.completed(fakeResponse);
        } else {
            StringBuilder sb =
                new StringBuilder(config.djfsConfig().uri().toASCIIString());
            sb.append(config.djfsConfig().firstCgiSeparator());
            sb.append("uid=");
            sb.append(context.prefix());
            QueryConstructor query = new QueryConstructor(sb);
            for (KaliRequestDoc doc: docsForUpdate) {
                query.append("resourceId", doc.resourceId());
            }
            AsyncClient client =
                djfsClient.adjust(context.session().context());
            try {
                client.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(
                            query.toString()),
                        new BasicHeader(
                            YandexHeaders.X_YA_SERVICE_TICKET,
                            djfsTvm2Ticket()),
                        new BasicHeader(
                            YandexHeaders.X_YANDEX_QUEUE_MESSAGE_ID,
                            context.zooQueue() + '@' + context.zooShardId()
                            + '@' + context.cgiZooQueueId())),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(client),
                    new UpstreamStaterFutureCallback<>(callback, djfsStater));
            } catch (URISyntaxException e) {
                throw new BadRequestException(e);
            }
        }
    }

    public int bodyTextLimit(final long user) {
        return personalIndexationLimits.tikaiteBodyTextLimit(user);
    }
}

