package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Level;

import org.apache.http.Header;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.disk.search.DiskCvField;
import ru.yandex.disk.search.DiskOcrField;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.string.URIParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.DiskIndexAction;
import ru.yandex.search.disk.DiskIndexActionParser;
import ru.yandex.util.string.HexStrings;

public class KaliRequestContext {
    public static final Header CHECK_DUPLICATE =
        HeaderUtils.createHeader(YandexHeaders.CHECK_DUPLICATE, "true");
    public static final int QUEUE_ID_PADDING = 16;

    private static final String PREFIX = "prefix";
    private static final String PRESERVEFIELDS = "PreserveFields";
    private static final String ACTION = "action";
    private static final String DOCS = "docs";
    private static final String ID = "id";
    private static final String RESOURCE_ID = "resource_id";
    private static final String TIMESTAMP = "timestamp";
    private static final int CALLBACK_NUMBER_PADDING = 8;
    private static final int INDEX_IO_PRIO = 3000;
    private static final int REINDEX_IO_PRIO = 5000;

    private static final List<JsonObject> OCR_CV_PRESERVE_FIELDS;

    static {
        JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
        for (DiskCvField field: DiskCvField.values()) {
            list.add(new JsonString(field.fieldName()));
        }

        for (DiskOcrField field: DiskOcrField.values()) {
            list.add(new JsonString(field.fieldName()));
        }

        OCR_CV_PRESERVE_FIELDS = Collections.unmodifiableList(list);
    }

    private final Kali kali;
    private final ProxySession session;
    private final long prefix;
    private final KaliActionType kaliActionType;
    private final DiskIndexAction actionType;
    private final KaliCleanupType cleanupType;
    private final Long timestamp;
    private final String zooQueue;
    private final long zooShardId;
    private final long zooQueueId;
    private final long cgiZooQueueId;
    private final AsyncClient searcherClient;
    private final AsyncClient indexerClient;
    private final List<URI> callbacks;
    private final List<URI> cvCallbacks;
    private final List<URI> ocrCallbacks;
    private final FaceSubprocessor faceProcessor;
    private final MultiFutureCallback<Object> callback;
    private ServerException registeredException = null;

    public KaliRequestContext(final Kali kali, final ProxySession session)
        throws BadRequestException
    {
        this.kali = kali;
        this.session = session;
        CgiParams params = session.params();
        prefix = params.get(PREFIX, NonNegativeLongValidator.INSTANCE);

        actionType = params.get(ACTION, DiskIndexActionParser.NON_EMPTY);
        switch (actionType) {
            case RM:
            case KICKED_BY_UNSHARE:
            case LEAVE_FOLDER:
            case TRASH_DROP_ALL:
            case TRASH_DROP_ELEMENT:
                kaliActionType = KaliActionType.REMOVE;
                break;
            case REINDEX:
                kaliActionType = KaliActionType.REINDEX;
                break;
            default:
                kaliActionType = KaliActionType.UPDATE;
                break;
        }
        cleanupType = params.getEnum(
            KaliCleanupType.class,
            "cleanup-type",
            KaliCleanupType.NONE);
        timestamp = params.get(
            TIMESTAMP,
            null,
            NonNegativeLongValidator.INSTANCE);
        zooQueue = session.headers().get(
            YandexHeaders.ZOO_QUEUE,
            NonEmptyValidator.INSTANCE);
        zooShardId = session.headers().get(
            YandexHeaders.ZOO_SHARD_ID,
            NonNegativeLongValidator.INSTANCE);
        zooQueueId = session.headers().getLong(
            YandexHeaders.ZOO_QUEUE_ID,
            Long.MIN_VALUE);
        // zooQueueId could absent for parallel requests, while this
        // value is taken from zoo-queue-id CGI-param
        if (zooQueueId < 0) {
            cgiZooQueueId = session.params().get(
                "zoo-queue-id",
                NonNegativeLongValidator.INSTANCE);
        } else {
            cgiZooQueueId = zooQueueId;
        }

        searcherClient =
            kali.searcherClient().adjust(session.context());
        indexerClient =
            kali.indexerClient().adjust(session.context());

        List<String> callbacks = params.getAll("callback");
        if (callbacks.isEmpty()) {
            this.callbacks = null;
        } else {
            this.callbacks = new ArrayList<>(callbacks.size() + 1);
            for (String callback: callbacks) {
                String effectiveUri;
                if (timestamp == null) {
                    effectiveUri = callback;
                } else {
                    char c;
                    if (callback.indexOf('?') == -1) {
                        c = '?';
                    } else {
                        c = '&';
                    }
                    effectiveUri = callback + c + "timestamp=" + timestamp;
                }
                try {
                    this.callbacks.add(
                        new URI(effectiveUri).parseServerAuthority());
                } catch (URISyntaxException e) {
                    throw new BadRequestException("Bad callback: " + callback);
                }
            }
        }
        cvCallbacks = params.getAll(
            "cv_callback",
            new ArrayList<>(),
            URIParser.INSTANCE,
            new ArrayList<>());
        ocrCallbacks = params.getAll(
            "ocr_callback",
            Collections.emptyList(),
            URIParser.INSTANCE,
            new ArrayList<>());


        FutureCallback<? super Object> callback =
            new LagCallback<>(this, timestamp);
        if (zooQueueId >= 0L) {
            callback = new CommitCallback(callback);
        }
        FutureCallback<? super List<Object>> listCallback = callback;

        if (this.callbacks != null) {
            listCallback = new CallbacksCallback(callback);
        }

        faceProcessor = FaceSubprocessor.createOrNull(this, listCallback);
        if (faceProcessor != null) {
            listCallback = faceProcessor;
            session.logger().info("Face subprocessor enabled");
        }

        listCallback = new ThrowExceptionCallback<>(this, listCallback);
        this.callback = new MultiFutureCallback<>(listCallback);
    }

    public Kali kali() {
        return kali;
    }

    public ProxySession session() {
        return session;
    }

    public long prefix() {
        return prefix;
    }

    public DiskIndexAction actionType() {
        return actionType;
    }

    public KaliActionType kaliActionType() {
        return kaliActionType;
    }

    public KaliCleanupType cleanupType() {
        return cleanupType;
    }

    public String zooQueue() {
        return zooQueue;
    }

    public long zooShardId() {
        return zooShardId;
    }

    public long cgiZooQueueId() {
        return cgiZooQueueId;
    }

    public MultiFutureCallback<Object> callback() {
        return callback;
    }

    public FaceSubprocessor faceProcessor() {
        return faceProcessor;
    }

    public void appendParams(
        final QueryConstructor query,
        final KaliRequestDoc doc)
        throws BadRequestException
    {
        query.append(PREFIX, prefix);
        query.append(ID, doc.id());
        query.append(RESOURCE_ID, doc.resourceId());
        query.append(ACTION, doc.kaliActionType().toString());
        if (timestamp != null) {
            query.append(TIMESTAMP, timestamp.longValue());
        }
    }

    public void prologue(
        final JsonWriterBase writer,
        final KaliRequestDoc doc)
        throws IOException
    {
        prologue(writer, doc, true);
    }

    public void prologue(
        final JsonWriterBase writer,
        final KaliRequestDoc doc,
        final boolean full)
        throws IOException
    {
        writer.startObject();
        writer.key(PREFIX);
        writer.value(prefix);
        writer.key(DOCS);
        writer.startArray();
        writer.startObject();
        writer.key(ID);
        writer.value(doc.id());
        // TODO: Change this on primary key switching
        if (full) {
            writer.key(RESOURCE_ID);
            writer.value(doc.resourceId());
            writer.key("owner");
            writer.value(doc.owner());
            if (timestamp != null) {
                writer.key(TIMESTAMP);
                writer.value(timestamp.longValue());
            }
        }
    }

    public void epilogue(final JsonWriterBase writer) throws IOException {
        epilogue(writer, false, false);
    }

    public void epilogue(
        final JsonWriterBase writer,
        final boolean preserveFields,
        final boolean addIfNotExits)
        throws IOException
    {
        writer.endObject();
        writer.endArray();
        if (preserveFields) {
            writer.key(PRESERVEFIELDS);
            writer.value(OCR_CV_PRESERVE_FIELDS);
        }

        if (addIfNotExits) {
            writer.key("AddIfNotExists");
            writer.value(true);
        }

        writer.endObject();
    }

    public AsyncClient searcherClient() {
        return searcherClient;
    }

    public AsyncClient indexerClient() {
        return indexerClient;
    }

    public List<URI> cvCallbacks() {
        return cvCallbacks;
    }

    public List<URI> ocrCallbacks() {
        return ocrCallbacks;
    }

    public int ioPrio() {
        if (kaliActionType == KaliActionType.REINDEX) {
            return REINDEX_IO_PRIO;
        } else {
            return INDEX_IO_PRIO;
        }
    }

    public ServerException registeredException() {
        return registeredException;
    }

    public void registerException(final ServerException e) {
        registeredException = e;
    }

    private class CallbacksCallback extends FilterFutureCallback<List<Object>> {
        CallbacksCallback(final FutureCallback<Object> callback) {
            super(callback);
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void completed(final List<Object> response) {
            //session.logger().info("Callbacks callback response" + response);
            JsonWriterBase writer = null;
            DecodableByteArrayOutputStream cbData = null;
            try {
                for (Object item: response) {
                    if (item == null) {
                        continue;
                    }

                    if (item instanceof JsonValue) {
                        if (cbData == null) {
                            cbData = new DecodableByteArrayOutputStream();
                            writer = JsonType.NORMAL.create(cbData);
                            writer.startArray();
                        }

                        writer.value((JsonValue) item);
                    }
                }

                if (cbData != null) {
                    writer.endArray();
                    writer.close();
                }
            } catch (IOException ioe) {
                session.logger().log(
                    Level.WARNING,
                    "Failed to create callback record",
                    ioe);
            }

            if (cbData != null) {
                session.logger().info(
                    "Data is not null, using post callback");

            }

            MultiFutureCallback<Object> multiCallback =
                new MultiFutureCallback<>(callback);
            List<Header> headers = new ArrayList<>(2 + 1);
            headers.add(
                HeaderUtils.createHeader(
                    YandexHeaders.SERVICE,
                    kali.config().callbacksQueue()));
            headers.add(
                HeaderUtils.createHeader(
                    YandexHeaders.ZOO_SHARD_ID,
                    Long.toString(zooShardId)));
            headers.add(CHECK_DUPLICATE);
            AsyncClient client =
                kali.callbacksClient().adjust(session.context());
            Supplier<? extends HttpClientContext> contextGenerator =
                session.listener().createContextGeneratorFor(client);
            // for photoslice queue id 42 this would generate hashes like
            // 70686f746f736c696365f000000000000002af00000001
            //          ^          ^                ^
            //          |          |                |
            // p h o t o s l i c e |                |
            //                     |                |
            //                     |                |
            //    queue id separator                callback number separator
            String queueNamePrefix = HexStrings.LOWER.toString(
                zooQueue.getBytes(StandardCharsets.UTF_8));
            StringBuilder hash = new StringBuilder(queueNamePrefix);
            hash.append('f');
            String zooQueueIdHex = Long.toHexString(cgiZooQueueId);
            for (int i = zooQueueIdHex.length(); i < QUEUE_ID_PADDING; ++i) {
                hash.append('0');
            }
            hash.append(zooQueueIdHex);
            hash.append('f');
            String hashPrefix = new String(hash);

            // face callback
            //
            for (int i = 0; i < callbacks.size(); ++i) {
                hash.setLength(0);
                hash.append(hashPrefix);
                String callbackNumberHex = Integer.toHexString(i);
                for (int j = callbackNumberHex.length();
                    j < CALLBACK_NUMBER_PADDING;
                    ++j)
                {
                    hash.append('0');
                }
                hash.append(callbackNumberHex);
                List<Header> requestHeaders =
                    new ArrayList<>(headers.size() + 1);
                requestHeaders.addAll(headers);
                requestHeaders.add(
                    HeaderUtils.createHeader(
                        YandexHeaders.ZOO_HASH,
                        new String(hash)));

                Supplier<HttpAsyncRequestProducer> reqSupplier;
                if (cbData != null) {
                    reqSupplier =
                        new AsyncPostURIRequestProducerSupplier(
                            callbacks.get(i),
                            cbData.toByteArray(),
                            ContentType.APPLICATION_JSON.withCharset(
                                StandardCharsets.UTF_8));
                } else {
                    URI callback = callbacks.get(i);
                    if (callback.toString().contains("set_dimensions")) {
                        session.logger().info(
                            "Skipping callback " + callback);
                        continue;
                    }
                    reqSupplier =
                        new AsyncGetURIRequestProducerSupplier(
                            callbacks.get(i));
                }

                client.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        reqSupplier,
                        requestHeaders),
                    EmptyAsyncConsumerFactory.OK,
                    contextGenerator,
                    new UpstreamStaterFutureCallback<>(
                        multiCallback.newCallback(),
                        kali.callbacksStater()));
            }
            multiCallback.done();
        }
    }

    // Should be used only if zooQueueId > 0
    private class CommitCallback extends FilterFutureCallback<Object> {
        CommitCallback(final FutureCallback<? super Object> callback) {
            super(callback);
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void completed(final Object response) {
            AsyncClient client =
                kali.indexerClient().adjustZooHeaders(session.context());
            if (timestamp != null) {
                client = client.addHeader(
                    YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                    Long.toString(TimeUnit.SECONDS.toMillis(timestamp)));
            }
            client.execute(
                kali.config().indexerConfig().host(),
                new BasicAsyncRequestProducerGenerator(
                    "/delete?commit",
                    "{$prefix\0:" + prefix + ",$docs\0:[]}",
                    kali.indexerContentType()),
                EmptyAsyncConsumerFactory.ANY_GOOD,
                session.listener().createContextGeneratorFor(client),
                callback);
        }
    }

    private static class LagCallback<T>
        extends AbstractProxySessionCallback<T>
    {
        private final Long timestamp;
        private final Kali kali;

        LagCallback(
            final KaliRequestContext context,
            final Long timestamp)
        {
            super(context.session);
            this.timestamp = timestamp;
            kali = context.kali;
        }

        @Override
        @SuppressWarnings("try")
        public void completed(final T response) {
            if (timestamp != null) {
                long now = TimeUnit.MILLISECONDS.toSeconds(
                    System.currentTimeMillis());
                long lag = now - timestamp;
                session.logger().info("Processing lag: " + lag + ' ' + 's');
                kali.lag(lag);
            }

            session.response(HttpStatus.SC_OK);
        }
    }

    private static class ThrowExceptionCallback<T>
        extends AbstractProxySessionCallback<T>
    {
        private final KaliRequestContext context;
        private final FutureCallback<T> callback;

        ThrowExceptionCallback(
            final KaliRequestContext context,
            final FutureCallback<T> callback)
        {
            super(context.session);
            this.context = context;
            this.callback = callback;
        }

        @Override
        @SuppressWarnings("try")
        public void completed(final T response) {
            if (context.registeredException() != null) {
                failed(context.registeredException());
            } else {
                callback.completed(response);
            }
        }
    }
}

