package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpException;

import ru.yandex.concurrent.AsyncLockHolder;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.util.AbstractFilterMultiFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.util.string.StringUtils;

public class ResourceCallback
    extends AbstractFilterMultiFutureCallback<JsonObject, Object>
{
    private final KaliRequestContext context;
    private final List<KaliRequestDoc> docsForRemoval;

    public ResourceCallback(
        final KaliRequestContext context,
        final List<KaliRequestDoc> docsForRemoval)
    {
        super(context.callback());
        this.context = context;
        this.docsForRemoval = docsForRemoval;
    }

    @Override
    public void completed(final JsonObject response) {
        try {
            completed(response.get("items").asList());
        } catch (HttpException e) {
            failed(e);
        } catch (IOException e) {
            failed(new ServiceUnavailableException(e));
        } catch (JsonException e) {
            failed(
                new UnsupportedMediaTypeException(
                    "Malformed documents meta: "
                    + JsonType.HUMAN_READABLE.toString(response),
                    e));
        }
    }

    private void completed(final JsonList docs)
        throws HttpException, IOException, JsonException
    {
        Kali kali = context.kali();
        if (docs.isEmpty() && docsForRemoval.isEmpty()) {
            kali.diskDocumentNotFound();
        } else {
            kali.diskDocumentFound();
        }
        String prefix = Long.toString(context.prefix()) + '#';
        try (GenericAutoCloseableHolder<
                RuntimeException,
                GenericAutoCloseableChain<RuntimeException>> chain =
                new GenericAutoCloseableHolder<>(
                    new GenericAutoCloseableChain<>()))
        {
            List<KaliTask> tasks = new ArrayList<>(docs.size());
            for (JsonObject object: docs) {
                JsonMap jsonDoc = object.asMap();
                DiskDocumentType type =
                    jsonDoc.getEnum(DiskDocumentType.class, "type");
                KaliDocumentContext doc =
                    new KaliDocumentContext(context, jsonDoc, type);
                AsyncLockHolder lock = new AsyncLockHolder(
                    context.kali().lockStorage(),
                    StringUtils.concat(prefix, doc.resourceId()));
                chain.get().add(lock);
                kali.accountUpdate();
                tasks.add(type.createIndexationTask(lock, doc));
                kali.documentType(type);
            }
            for (KaliRequestDoc doc: docsForRemoval) {
                kali.accountRemoval();
                tasks.add(kali.createDeleteDocTask(context, doc, chain.get()));
            }
            callback.done();
            for (KaliTask task: tasks) {
                task.execute();
            }
            chain.release();
        }
    }
}

