package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.proxy.RequestProxy;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;

public class SearchBackendProxyWithCallbacksHandler
    implements ProxyRequestHandler
{
    private static final int CALLBACK_NUMBER_PADDING = 4;
    private final Kali kali;
    private final RequestProxy requestProxy;
    private final List<HttpHost> hosts;

    private static final Header CHECK_DUPLICATE =
        HeaderUtils.createHeader(YandexHeaders.CHECK_DUPLICATE, "true");

    public SearchBackendProxyWithCallbacksHandler(
        final Kali kali)
    {
        this.kali = kali;

        this.requestProxy = new RequestProxy(kali.indexerClient());
        this.hosts =
            Collections.singletonList(
                kali.config().indexerConfig().host());
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        List<String> callbacks = session.params().getAll("callback");
        if (callbacks.isEmpty()) {
            requestProxy.handle(session, hosts);
            return;
        }

        Context context = new Context(session, callbacks);
        requestProxy.handle(
            session,
            hosts,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            new Callback(context));
    }

    private class Callback
        extends AbstractProxySessionCallback<HttpResponse>
    {
        private final Context context;

        public Callback(
            final Context context)
        {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final HttpResponse response) {
            AsyncClient client =
                kali.callbacksClient().adjust(session.context());
            Supplier<? extends HttpClientContext> contextGenerator =
                session.listener().createContextGeneratorFor(client);

            MultiFutureCallback<Object> multiCallback =
                new MultiFutureCallback<>(
                    new CallbacksCallback(session, response));

            List<Header> headers = new ArrayList<>(2 + 1);
            headers.add(
                HeaderUtils.createHeader(
                    YandexHeaders.SERVICE,
                    kali.config().callbacksQueue()));

            headers.add(
                HeaderUtils.createHeader(
                    YandexHeaders.ZOO_SHARD_ID,
                    Long.toString(context.zooShardId())));
            headers.add(CHECK_DUPLICATE);

            StringBuilder hash =
                new StringBuilder(context.zooHash().length() + 1);

            for (int i = 0; i < context.callbacks().size(); ++i) {
                hash.setLength(0);
                hash.append(context.zooHash());
                String callbackNumberHex = Integer.toHexString(i);
                for (int j = callbackNumberHex.length();
                     j < CALLBACK_NUMBER_PADDING;
                     ++j)
                {
                    hash.append('0');
                }
                hash.append(callbackNumberHex);
                List<Header> requestHeaders =
                    new ArrayList<>(headers.size() + 1);
                requestHeaders.addAll(headers);
                requestHeaders.add(
                    HeaderUtils.createHeader(
                        YandexHeaders.ZOO_HASH,
                        new String(hash)));

                Supplier<HttpAsyncRequestProducer> reqSupplier;
                reqSupplier =
                    new AsyncGetURIRequestProducerSupplier(
                        context.callbacks().get(i));

                client.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        reqSupplier,
                        requestHeaders),
                    EmptyAsyncConsumerFactory.OK,
                    contextGenerator,
                    new UpstreamStaterFutureCallback<>(
                        multiCallback.newCallback(),
                        kali.callbacksStater()));
            }
            multiCallback.done();
        }
    }

    private static class CallbacksCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final HttpResponse response;

        public CallbacksCallback(
            final ProxySession session,
            final HttpResponse response)
        {
            super(session);
            this.response = response;
        }

        @Override
        public void completed(final Object o) {
            session.response(response);
        }
    }

    private static class Context {
        private final ProxySession session;
        private final List<URI> callbacks;
        private final String zooQueue;
        private final Long zooQueueId;
        private final Long zooShardId;
        private final String zooHash;
        private final Long prefix;

        public Context(
            final ProxySession session,
            final List<String> callbacks)
            throws HttpException
        {
            this.session = session;

            zooHash =
                session.params().getString("zoo_hash");
            prefix =
                session.params().getLong("prefix");

            this.callbacks = new ArrayList<>(callbacks.size());
            for (String callback: callbacks) {
                try {
                    URI uri = new URI(callback).parseServerAuthority();
                    HttpHost host = URIUtils.extractHost(uri);
                    if (host == null) {
                        throw new BadRequestException("Bad callback: no host:" + uri);
                    }
                    this.callbacks.add(uri);
                } catch (URISyntaxException e) {
                    throw new BadRequestException("Bad callback: " + callback);
                }
            }

            zooQueue = session.headers().get(
                YandexHeaders.ZOO_QUEUE,
                NonEmptyValidator.INSTANCE);
            zooShardId = session.headers().get(
                YandexHeaders.ZOO_SHARD_ID,
                NonNegativeLongValidator.INSTANCE);
            zooQueueId = session.headers().getLong(
                YandexHeaders.ZOO_QUEUE_ID,
                Long.MIN_VALUE);
        }

        public ProxySession session() {
            return session;
        }

        public List<URI> callbacks() {
            return callbacks;
        }

        public String zooQueue() {
            return zooQueue;
        }

        public Long zooQueueId() {
            return zooQueueId;
        }

        public Long zooShardId() {
            return zooShardId;
        }

        public String zooHash() {
            return zooHash;
        }

        public Long prefix() {
            return prefix;
        }
    }
}
