package ru.yandex.search.disk.kali;

import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.disk.search.DiskField;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.PassPayloadThroughFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;

public class TikaiteCallback
    extends AbstractFilterFutureCallback<JsonObject, Object>
{
    private final KaliDocumentContext doc;
    private final KaliRequestContext context;

    public TikaiteCallback(
        final FutureCallback<Object> callback,
        final KaliDocumentContext doc)
    {
        super(callback);
        this.doc = doc;
        context = doc.requestContext();
    }

    @Override
    public void failed(final Exception e) {
        if (RequestErrorType.temporaryFailure(e)
            || !AsyncClient.serverUnavailalbe(e))
        {
            super.failed(e);
        } else {
            context.kali().tikaiteFailed();
            context.session().logger().log(
                Level.WARNING,
                "Failed to retrieve data from tikaite, skipping document: "
                + context.session().listener().details(),
                e);
            StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            JsonObject fakeResponse = JsonMap.singletonMap(
                "tikaite_error",
                new JsonString(sbw.toString()));
            completedInternal(fakeResponse);
        }
    }

    @Override
    public void completed(final JsonObject response) {
        context.kali().tikaiteCompleted();
        completedInternal(response);
    }

    private void completedInternal(final JsonObject response) {
        try {
            JsonMap map;
            try {
                map = response.asMap();
            } catch (JsonException e) {
                throw new ServiceUnavailableException(
                    "Malformed tikaite response: "
                    + JsonType.HUMAN_READABLE.toString(response),
                    e);
            }
            completedInternal(map);
        } catch (HttpException e) {
            super.failed(e);
        } catch (IOException e) {
            super.failed(new ServiceUnavailableException(e));
        }
    }

    private void completedInternal(final JsonMap response)
        throws HttpException, IOException
    {
        JsonObject mimetype = response.remove(DiskField.MIMETYPE.fieldName());
        AbstractMedia media = null;
        boolean image;
        boolean video;
        if (mimetype == JsonNull.INSTANCE) {
            image = false;
            video = false;
        } else {
            response.put("tikaite_mimetype", mimetype);
            try {
                image = mimetype.asString().startsWith("image/") ;
                video = mimetype.asString().startsWith("video/") ;
                if (image || video) {
                    int imageWidth = response.getInt("width", -1);
                    int imageHeight = response.getInt("height", -1);
                    Double longitude = response.getDouble("longitude", null);
                    Double latitude = response.getDouble("latitude", null);
                    Double altitude = response.getDouble("altitude", null);
                    String orientation =
                        response.getString("orientation", "unknown");
                    if (video) {
                        media = new Video(
                            doc.id(),
                            imageWidth,
                            imageHeight,
                            latitude,
                            longitude,
                            altitude,
                            orientation);
                    } else if (image) {
                        media = new Image(
                            doc,
                            imageWidth,
                            imageHeight,
                            latitude,
                            longitude,
                            altitude,
                            orientation);
                    }
                }
            } catch (JsonException e) {
                throw new ServiceUnavailableException(
                    "Malformed tikaite mimetype: "
                    + JsonType.HUMAN_READABLE.toString(response),
                    e);
            }
        }
        Kali kali = context.kali();
        if (image) {
            kali.imageModification();
        } else {
            kali.fileModification();
        }
        DiskDocumentType type = doc.type();
        StringBuilderWriter sbw = new StringBuilderWriter();

        try (JsonWriter writer = new DollarJsonWriter(sbw)) {
            context.prologue(writer, doc);

            type.writeDocument(
                doc.doc(),
                writer,
                new DocumentErrorHandler(context));
            for (Map.Entry<String, JsonObject> entry: response.entrySet()) {
                writer.key(entry.getKey());
                entry.getValue().writeValue(writer);
            }
            context.epilogue(writer, true, true);
        }

        QueryConstructor query = new QueryConstructor("/update?file");
        context.appendParams(query, doc);

        FutureCallback<Object> callback;
        if (image) {
            callback = new ImageCallback(this.callback, media, doc);
        } else if (video) {
            callback =
                new PassPayloadThroughFutureCallback<>(
                    media,
                    this.callback);
        } else {
            callback = new PassPayloadThroughFutureCallback<>(doc, this.callback);
        }
        context.indexerClient().execute(
            kali.config().indexerConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                query.toString(),
                sbw.toString(),
                kali.indexerContentType()),
            EmptyAsyncConsumerFactory.ANY_GOOD,
            context.session().listener().createContextGeneratorFor(
                context.indexerClient()),
            callback);
    }
}

