package ru.yandex.ocr.proxy;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.disk.search.DiskCvField;
import ru.yandex.disk.search.DiskField;
import ru.yandex.disk.search.DiskParams;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

public class CvHandler implements ProxyRequestHandler {
    private static final String STID = "stid";
    private static final String PREFIX = "prefix";
    private static final String ID = "id";
    private static final String FACES = "faces";
    private static final String X = "x";
    private static final String Y = "y";
    private static final String WIDTH = "width";
    private static final String HEIGHT = "height";
    private static final String CLASSES = "classes";

    private final OcrProxy proxy;

    public CvHandler(final OcrProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        CgiParams params = session.params();
        String stid = params.get(STID, null, NonEmptyValidator.INSTANCE);
        String previewStid = params.getString(DiskParams.PREVIEW_STID, null);
        if (stid == null) {
            long prefix =
                params.get(PREFIX, NonNegativeLongValidator.INSTANCE);
            String id = params.get(ID, NonEmptyValidator.INSTANCE);
            QueryConstructor query = new QueryConstructor(
                "/search-stid?get=stid,preview_stid"
                    + "&postfilter=mediatype+%3d%3d+9&json-type=dollar");
            query.append(PREFIX, prefix);
            query.append("text", "id:" + id);
            AsyncClient client =
                proxy.searchClient().adjust(session.context());
            client.execute(
                proxy.config().searchConfig().host(),
                new BasicAsyncRequestProducerGenerator(query.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new StidCallback(proxy, session));
        } else {
            handle(new OcrProxyContext(proxy, session, stid, previewStid));
        }
    }

    private static void handle(final OcrProxyContext context)
        throws HttpException
    {
        OcrProxy proxy = context.proxy();
        QueryConstructor query =
            new QueryConstructor("/process/handler?", false);
        query.append(STID, context.unistorageKey());
        query.append("passcache", "1");
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(query.toString());
        context.adjustSrwHeaders(producerGenerator);
        AsyncClient client =
            proxy.imageparserClient().adjust(context.proxySession().context());
        client.execute(
            proxy.config().imageparserConfig().host(),
            producerGenerator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.proxySession().listener().createContextGeneratorFor(
                client),
            new ImageparserCallback(context));
    }

    @Override
    public String toString() {
        return "Requests CV for specified stid and index it";
    }

    private static class StidCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final OcrProxy proxy;

        StidCallback(final OcrProxy proxy, final ProxySession session) {
            super(session);
            this.proxy = proxy;
        }

        @Override
        public void completed(final JsonObject root) {
            try {
                JsonList docs = root.get("hitsArray").asList();
                String stid;
                if (docs.isEmpty()) {
                    stid = null;
                } else {
                    stid = docs.get(0).asMap().get(
                        STID,
                        null,
                        NonEmptyValidator.INSTANCE);
                }
                if (stid == null) {
                    session.logger().info("No suitable document found");
                    session.response(HttpStatus.SC_OK);
                } else {
                    JsonMap doc = docs.get(0).asMap();
                    handle(
                        new OcrProxyContext(
                            proxy,
                            session,
                            doc.get(STID, NonEmptyValidator.INSTANCE),
                            doc.getString(
                                DiskField.PREVIEW_STID.fieldName(),
                                null)));
                }
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }

    private static class ImageparserCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final OcrProxyContext context;

        ImageparserCallback(final OcrProxyContext context) {
            super(context.proxySession());
            this.context = context;
        }

        @Override
        public void failed(final Exception e) {
            CvStat stat = new CvStat();
            RequestErrorType errorType =
                RequestErrorType.ERROR_CLASSIFIER.apply(e);
            if (errorType == RequestErrorType.NON_RETRIABLE
                || errorType == RequestErrorType.HOST_NON_RETRIABLE)
            {
                if (OcrProxy.criticalNonRetriable(e, stat)) {
                    // Something very bad happended here. Rate limit or
                    // authorization error, let the caller decide what to do.
                    super.failed(e);
                } else {
                    // This file is unprocessable, error type already accounted
                    // in stat, so save stat and reply 200 OK
                    context.proxy().stat(stat);
                    session.response(HttpStatus.SC_OK);
                }
            } else {
                // Imageparser can't process image after all retries, skip it
                stat.error(true);
                context.proxy().stat(stat);
                session.response(HttpStatus.SC_OK);
            }
        }

        protected String buildCallbackData(
            final OcrProxyContext context,
            final JsonMap response)
            throws IOException, JsonException
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key("uid");
                writer.value(context.prefix());
                writer.key(ID);
                writer.value(context.id());
                writer.key(STID);
                writer.value(context.stid());
                writer.key(WIDTH);
                writer.value(context.imageWidth());
                writer.key(HEIGHT);
                writer.value(context.imageHeight());

                JsonMap classes = response.getMapOrNull(CLASSES);
                if (classes != null) {
                    writeClasses(writer, classes, false);
                }

                writer.endObject();
            }

            return sbw.toString();
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonMap cvResult = response.asMap();
                String i2t = cvResult.getOrNull("i2t_hex");
                JsonList faces = cvResult.getListOrNull(FACES);
                JsonMap classes = cvResult.getMapOrNull(CLASSES);
                CvStat stat = new CvStat();

                String callbackData = "";
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                    writer.startObject();
                    writer.key(PREFIX);
                    writer.value(context.prefix());
                    writer.key("query");
                    writer.value(OcrProxy.luceneUpdateUri(context));
                    writer.key("docs");
                    writer.startArray();
                    writer.startObject();
                    writer.key(DiskCvField.I2T_KEYWORD.fieldName());
                    if (i2t != null) {
                        stat.hasI2T(true);
                        writer.value(i2t);
                    } else {
                        writer.value(JsonNull.INSTANCE);
                    }

                    if (faces != null && !faces.isEmpty()) {
                        stat.hasFaces(true);
                        StringBuilder widths = new StringBuilder();
                        StringBuilder heights = new StringBuilder();
                        StringBuilderWriter facesWriter =
                            new StringBuilderWriter();
                        try (JsonWriter facesJsonWriter =
                                new JsonWriter(facesWriter))
                        {
                            facesJsonWriter.startArray();
                            for (JsonObject faceObject: faces) {
                                JsonMap face = faceObject.asMap();
                                String x = face.getString(X);
                                String y = face.getString(Y);
                                String width = face.getString(WIDTH);
                                String height = face.getString(HEIGHT);
                                facesJsonWriter.startObject();
                                facesJsonWriter.key(X);
                                facesJsonWriter.value(x);
                                facesJsonWriter.key(Y);
                                facesJsonWriter.value(y);
                                facesJsonWriter.key(WIDTH);
                                facesJsonWriter.value(width);
                                facesJsonWriter.key(HEIGHT);
                                facesJsonWriter.value(height);
                                facesJsonWriter.endObject();
                                widths.append(width);
                                widths.append('\n');
                                heights.append(height);
                                heights.append('\n');
                            }
                            facesJsonWriter.endArray();
                        }
                        writer.key(DiskCvField.FACES.fieldName());
                        writer.value(facesWriter.toString());
                        writer.key(DiskCvField.FACES_WIDTHS.fieldName());
                        writer.value(widths);
                        writer.key(DiskCvField.FACES_HEIGHTS.fieldName());
                        writer.value(heights);
                    } else {
                        writer.key(DiskCvField.FACES.fieldName());
                        writer.value(JsonNull.INSTANCE);
                        writer.key(DiskCvField.FACES_WIDTHS.fieldName());
                        writer.value(JsonNull.INSTANCE);
                        writer.key(DiskCvField.FACES_HEIGHTS.fieldName());
                        writer.value(JsonNull.INSTANCE);
                    }

                    if (classes == null) {
                        classes = JsonMap.EMPTY;
                    }

                    boolean hasClasses = writeClasses(writer, classes, true);

                    stat.hasClasses(hasClasses);

                    writer.endObject();
                    writer.endArray();
                    writer.endObject();

                    if (context.hasCallbacks()) {
                        callbackData = buildCallbackData(context, cvResult);
                    }
                } catch (IOException e) {
                    throw new ServiceUnavailableException(e);
                }
                OcrProxy proxy = context.proxy();
                ImmutableOcrProxyConfig config = proxy.config();
                if (!stat.hasI2T() && !stat.hasFaces() && !stat.hasClasses()) {
                    long lag = context.lag();
                    stat.lag(lag);
                    proxy.stat(stat);

                    session.logger().info(
                        "Nothing extracted, lag = " + lag + ' ' + 's');

                    if (!context.faceCallbacks().isEmpty()) {
                        context.spawnCallbacks(
                            proxy.faceCallbacksClient(),
                            callbackData,
                            config.faceCallbacksConfig().requestCharset(),
                            config.faceCallbacksQueue(),
                            true,
                            OcrProxyContext.FACE_HASH_PREFIX,
                            context.faceCallbacks(),
                            new BasicProxySessionCallback(session, HttpStatus.SC_OK));
                    } else {
                        session.response(HttpStatus.SC_OK);
                    }

                    return;
                }

                QueryConstructor query = new QueryConstructor(
                    "/update?cv&prefix=" + context.prefix());
                query.append(ID, context.id());
                BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator(
                        query.toString(),
                        sbw.toString(),
                        ContentType.APPLICATION_JSON.withCharset(
                            config.indexerConfig().requestCharset()));
                Long timestamp = context.timestamp();
                if (timestamp != null) {
                    String queueName = config.cvQueue();
                    if ("/cv-reindex".equals(session.uri().path().decode())) {
                        queueName = queueName + "-reindex";
                    }
                    producerGenerator.addHeader(
                        YandexHeaders.ZOO_QUEUE,
                        queueName);
                    producerGenerator.addHeader(
                        YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                        Long.toString(TimeUnit.SECONDS.toMillis(timestamp)));
                    producerGenerator.addHeader(context.zooShardId());
                }
                AsyncClient client =
                    proxy.indexerClient().adjust(session.context());

                FutureCallback<Object> indexCallback =
                    new IndexerCallback(context, stat);

                if (!context.faceCallbacks().isEmpty()) {
                    indexCallback = context.spawnCallbacks(
                        proxy.faceCallbacksClient(),
                        callbackData,
                        config.faceCallbacksConfig().requestCharset(),
                        config.faceCallbacksQueue(),
                        true,
                        OcrProxyContext.FACE_HASH_PREFIX,
                        context.faceCallbacks(),
                        indexCallback);
                }

                if (!context.callbacks().isEmpty()) {
                    indexCallback = context.spawnCallbacks(
                        proxy.cvCallbacksClient(),
                        callbackData,
                        config.cvCallbacksConfig().requestCharset(),
                        config.cvCallbacksQueue(),
                        false,
                        OcrProxyContext.CV_HASH_PREFIX,
                        context.callbacks(),
                        indexCallback);
                }
                client.execute(
                    config.indexerConfig().host(),
                    producerGenerator,
                    EmptyAsyncConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    indexCallback);
            } catch (HttpException | JsonException e) {
                super.failed(e);
            }
        }
    }

    protected static boolean writeClasses(
        final JsonWriter writer,
        final JsonMap classes,
        final boolean writeNulls)
        throws IOException
    {
        boolean hasClasses = false;
        for (DiskCvField clazz: DiskCvField.CLASSES) {
            JsonObject probability =
                classes.get(clazz.fieldName());
            if (probability == JsonNull.INSTANCE) {
                if (writeNulls) {
                    writer.key(clazz.fieldName());
                    writer.value(probability);
                }
            } else {
                hasClasses = true;
                writer.key(clazz.fieldName());
                writer.value(probability);
            }
        }

        return hasClasses;
    }

    private static class IndexerCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final OcrProxyContext context;
        private final CvStat stat;

        IndexerCallback(final OcrProxyContext context, final CvStat stat) {
            super(context.proxySession());
            this.context = context;
            this.stat = stat;
        }

        @Override
        public void completed(final Object result) {
            long lag = context.lag();
            stat.lag(lag);
            context.proxy().stat(stat);
            context.proxySession().logger().info("CV lag: " + lag + ' ' + 's');
            context.proxySession().response(HttpStatus.SC_OK);
        }
    }
}

