package ru.yandex.ocr.proxy;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;

import ru.yandex.disk.search.DiskOcrField;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;

public class OcrHandler implements ProxyRequestHandler {
    private static final String STID = "stid";
    private static final String ID = "id";

    private final OcrProxy proxy;

    public OcrHandler(final OcrProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        OcrProxyContext context = new OcrProxyContext(proxy, session);
        QueryConstructor query =
            new QueryConstructor("/process/handler?", false);

        query.append(STID, context.unistorageKey());
        query.append("passcache", "1");
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(query.toString());

        context.adjustSrwHeaders(producerGenerator);
        AsyncClient client =
            proxy.ocraasClient().adjust(session.context());
        client.execute(
            proxy.config().ocraasConfig().host(),
            producerGenerator,
            AsyncStringConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new OcrCallback(context));
    }

    @Override
    public String toString() {
        return "Requests OCR for specified stid and index it";
    }

    private static class OcrCallback
        extends AbstractProxySessionCallback<String>
    {
        private final OcrProxyContext context;

        OcrCallback(final OcrProxyContext context) {
            super(context.proxySession());
            this.context = context;
        }

        @Override
        public void failed(final Exception e) {
            OcrStat stat = new OcrStat();
            RequestErrorType errorType =
                RequestErrorType.ERROR_CLASSIFIER.apply(e);
            if (errorType == RequestErrorType.NON_RETRIABLE
                || errorType == RequestErrorType.HOST_NON_RETRIABLE)
            {
                if (OcrProxy.criticalNonRetriable(e, stat)) {
                    // Something very bad happended here. Rate limit or
                    // authorization error, let the caller decide what to do.
                    super.failed(e);
                } else {
                    // This file is unprocessable, error type already accounted
                    // in stat, so save stat and reply 200 OK
                    context.proxy().stat(stat);
                    session.response(HttpStatus.SC_OK);
                }
            } else {
                // Imageparser can't process image after all retries, skip it
                stat.error(true);
                context.proxy().stat(stat);
                session.response(HttpStatus.SC_OK);
            }
        }

        @Override
        public void completed(final String result) {
            String text = result.trim();
            OcrProxy proxy = context.proxy();
            if (text.isEmpty() && !context.updateOnEmpty()) {
                proxy.stat(new OcrStat());
                session.response(HttpStatus.SC_OK);
            } else {
                long prefix = context.prefix();
                try {
                    StringBuilderWriter sbw = new StringBuilderWriter();
                    try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                        writer.startObject();
                        writer.key("prefix");
                        writer.value(prefix);
                        writer.key("query");
                        writer.value(OcrProxy.luceneUpdateUri(context));
                        writer.key("docs");
                        writer.startArray();
                        writer.startObject();
                        writer.key(DiskOcrField.OCR_TEXT.fieldName());
                        writer.value(text);
                        writer.endObject();
                        writer.endArray();
                        writer.endObject();
                    } catch (IOException e) {
                        throw new ServiceUnavailableException(e);
                    }
                    String body = sbw.toString();
                    QueryConstructor query =
                        new QueryConstructor("/update?ocr&prefix=" + prefix);
                    query.append(ID, context.id());
                    ImmutableOcrProxyConfig config = proxy.config();
                    BasicAsyncRequestProducerGenerator producerGenerator =
                        new BasicAsyncRequestProducerGenerator(
                            query.toString(),
                            body,
                            ContentType.APPLICATION_JSON.withCharset(
                                config.indexerConfig().requestCharset()));
                    Long timestamp = context.timestamp();
                    if (timestamp != null) {
                        producerGenerator.addHeader(
                            YandexHeaders.ZOO_QUEUE,
                            config.ocrQueue());
                        producerGenerator.addHeader(
                            YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                            Long.toString(
                                TimeUnit.SECONDS.toMillis(timestamp)));
                        producerGenerator.addHeader(context.zooShardId());
                    }
                    AsyncClient indexerClient =
                        proxy.indexerClient().adjust(session.context());
                    indexerClient.execute(
                        config.indexerConfig().host(),
                        producerGenerator,
                        EmptyAsyncConsumerFactory.OK,
                        session.listener().createContextGeneratorFor(
                            indexerClient),

                        // TODO different hash prefix ?
                        context.spawnCallbacks(
                            proxy.ocrCallbacksClient(),
                            body,
                            config.ocrCallbacksConfig().requestCharset(),
                            config.ocrCallbacksQueue(),
                            false,
                            OcrProxyContext.CV_HASH_PREFIX,
                            context.callbacks(),
                            new IndexerCallback(context, text)));
                } catch (HttpException e) {
                    failed(e);
                }
            }
        }
    }

    private static class IndexerCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final OcrProxyContext context;
        private final String text;

        IndexerCallback(final OcrProxyContext context, final String text) {
            super(context.proxySession());
            this.context = context;
            this.text = text;
        }

        @Override
        public void completed(final Object result) {
            long lag = context.lag();
            OcrStat stat = new OcrStat();
            stat.lag(lag);
            stat.ocrTextLength(text.length());
            stat.preview(context.isPreviewAvaStid());
            context.proxy().stat(stat);
            context.proxySession().logger().info(
                "OCR lag: " + lag + ' ' + 's');
            context.proxySession().response(HttpStatus.SC_OK);
        }
    }
}

