package ru.yandex.ocr.proxy;

import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;

import org.apache.http.HttpException;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.disk.search.DiskField;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.parser.JsonException;
import ru.yandex.search.request.util.SearchRequestText;

public class OcrProxy extends HttpProxy<ImmutableOcrProxyConfig> {
    private static final String AND = " AND ";

    private static final String STID_FIELD = DiskField.STID.fieldName() + ':';
    private static final String ID_FIELD = "id:";

    private static final int UPDATE_QUERY_SB_SIZE =
        STID_FIELD.length() + ID_FIELD.length() + AND.length();

    private final TimeFrameQueue<OcrStat> ocrStats;
    private final TimeFrameQueue<CvStat> cvStats;
    private final AsyncClient ocraasClient;
    private final AsyncClient imageparserClient;
    private final AsyncClient indexerClient;
    private final AsyncClient searchClient;
    private final AsyncClient ocrCallbacksClient;
    private final AsyncClient cvCallbacksClient;
    private final AsyncClient faceCallbacksClient;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;

    public OcrProxy(final ImmutableOcrProxyConfig config)
        throws GeneralSecurityException,
            HttpException,
            IOException,
            JsonException,
            URISyntaxException
    {
        super(config);
        ocrStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        cvStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        ocraasClient = client("OCRaaS", config.ocraasConfig());
        imageparserClient =
            client("Imageparser", config.imageparserConfig());
        indexerClient = client("Indexer", config.indexerConfig());
        searchClient = client("Search", config.searchConfig());
        ocrCallbacksClient =
            client("OCR-Callbacks", config.ocrCallbacksConfig());
        cvCallbacksClient =
            client("CV-Callbacks", config.cvCallbacksConfig());
        faceCallbacksClient =
            client("Face-Callbacks", config.faceCallbacksConfig());
        tvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("tvm2"),
            serviceContextRenewalTask,
            config.tvm2ClientConfig());
        register(new Pattern<>("/ocr", true), new OcrHandler(this));
        register(new Pattern<>("/cv", true), new CvHandler(this));
        registerStater(new OcrStater(ocrStats));
        registerStater(new CvStater(cvStats));
    }

    public static boolean criticalNonRetriable(
        final Exception e,
        final OcrProxyBasicStat stat)
    {
        boolean result = true;
        if (e instanceof BadResponseException) {
            switch (((BadResponseException) e).statusCode()) {
                case YandexHttpStatus.SC_NOT_FOUND:
                    stat.notFound(true);
                    result = false;
                    break;
                case YandexHttpStatus.SC_LOCKED:
                    stat.locked(true);
                    result = false;
                    break;
                case YandexHttpStatus.SC_UNSUPPORTED_MEDIA_TYPE:
                    stat.unsupported(true);
                    result = false;
                    break;
                case YandexHttpStatus.SC_REQUEST_TOO_LONG:
                    stat.tooLarge(true);
                    result = false;
                    break;
                default:
                    break;
            }
        }
        return result;
    }

    public void stat(final OcrStat stat) {
        ocrStats.accept(stat);
    }

    public void stat(final CvStat stat) {
        cvStats.accept(stat);
    }

    public AsyncClient ocraasClient() {
        return ocraasClient;
    }

    public AsyncClient imageparserClient() {
        return imageparserClient;
    }

    public AsyncClient indexerClient() {
        return indexerClient;
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public AsyncClient ocrCallbacksClient() {
        return ocrCallbacksClient;
    }

    public AsyncClient cvCallbacksClient() {
        return cvCallbacksClient;
    }

    public AsyncClient faceCallbacksClient() {
        return faceCallbacksClient;
    }

    @Override
    public void start() throws IOException {
        tvm2RenewalTask.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        super.close();
        tvm2RenewalTask.cancel();
    }

    public String apeTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.apeTvmClientId());
    }

    public String unistorageTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.unistorageTvmClientId());
    }

    public static StringBuilder luceneUpdateUri(final OcrProxyContext context) {
        // extra 2 chars for escaping
        StringBuilder updateQuery =
            new StringBuilder(
                UPDATE_QUERY_SB_SIZE
                    + context.stid().length() + 2
                    + context.id().length() + 2);
        updateQuery.append(ID_FIELD);
        updateQuery.append(
            SearchRequestText.fullEscape(context.id(), false));
        updateQuery.append(AND);
        updateQuery.append(STID_FIELD);
        updateQuery.append(
            SearchRequestText.fullEscape(context.stid(), false));
        return updateQuery;
    }
}

