package ru.yandex.ocr.proxy;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.disk.search.AvatarSrwKeyConverter;
import ru.yandex.disk.search.DiskParams;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.StringUtils;

public class OcrProxyContext {
    public static final String CV_HASH_PREFIX =
        HexStrings.LOWER.toString("cv".getBytes(StandardCharsets.UTF_8));
    public static final String FACE_HASH_PREFIX =
        HexStrings.LOWER.toString("face".getBytes(StandardCharsets.UTF_8));

    public static final String AVASTID_POSTFIX = "1280_nocrop/webp";
    private static final int PREFIX_PADDING = 16;
    private static final Header CHECK_DUPLICATE =
        HeaderUtils.createHeader(YandexHeaders.CHECK_DUPLICATE, "true");

    private static final long SHARDS = 65534L;
    private static final int QUEUE_ID_PADDING = 16;

    private final OcrProxy proxy;
    private final ProxySession proxySession;
    private final String stid;
    private final String processStid;
    private final String unistorageKey;
    private final boolean avaKey;
    private final boolean updateOnEmpty;
    private final long prefix;
    private final String id;
    private final Long timestamp;
    private final Long cgiZooQueueId;
    private final List<URI> callbacks;
    private final List<URI> faceCallbacks;
    private final Header zooShardId;
    private final Integer imageWidth;
    private final Integer imageHeight;

    public OcrProxyContext(
        final OcrProxy proxy,
        final ProxySession session)
        throws BadRequestException
    {
        this(
            proxy,
            session,
            session.params().get("stid", NonEmptyValidator.INSTANCE),
            session.params().getString(DiskParams.PREVIEW_STID, null));
    }

    // CSOFF: ParameterNumber
    public OcrProxyContext(
        final OcrProxy proxy,
        final ProxySession proxySession,
        final String stid,
        final String previewStid)
        throws BadRequestException
    {
        this.proxy = proxy;
        this.proxySession = proxySession;
        this.stid = stid;
        CgiParams params = proxySession.params();
        prefix = params.get("prefix", NonNegativeLongValidator.INSTANCE);
        id = params.get("id", NonEmptyValidator.INSTANCE);
        imageHeight = params.getInt("height", -1);
        imageWidth = params.getInt("width", -1);
        cgiZooQueueId = params.get(
            "zoo-queue-id",
            null,
            NonNegativeLongValidator.INSTANCE);
        if (!proxy.config().usePreviewStid() || previewStid == null) {
            this.unistorageKey = stid;
            this.processStid = stid;
            this.avaKey = false;
        } else {
            avaKey = AvatarSrwKeyConverter.INSTANCE.isAvatarStid(previewStid);
            if (avaKey) {
                this.processStid =
                    AvatarSrwKeyConverter.INSTANCE.parse(previewStid);
                this.unistorageKey = StringUtils.concat(
                    DiskParams.AVATAR_X_SRW_NAMESPACE.getValue(),
                    '/',
                    processStid,
                    '/',
                    AVASTID_POSTFIX);
            } else {
                this.unistorageKey = previewStid;
                this.processStid = previewStid;
            }
        }

        // should we send update if nothing extracted
        updateOnEmpty =
            params.getBoolean("update-empty", false);

        timestamp = params.getLong("timestamp", null);
        this.callbacks = parseCallbacks(params, "callback");
        this.faceCallbacks = parseCallbacks(params, "face_callback");
        zooShardId = HeaderUtils.createHeader(
            YandexHeaders.ZOO_SHARD_ID,
            Long.toString(prefix % SHARDS));
    }
    // CSON: ParameterNumber

    private List<URI> parseCallbacks(
        final CgiParams params,
        final String paramName)
        throws BadRequestException
    {
        List<String> callbacks = params.getAll(paramName);
        if (callbacks.isEmpty()) {
            return Collections.emptyList();
        }

        List<URI> parsedCallbacks = new ArrayList<>(callbacks.size());
        for (String callback: callbacks) {
            String effectiveUri;
            if (timestamp == null) {
                effectiveUri = callback;
            } else {
                char c;
                if (callback.indexOf('?') == -1) {
                    c = '?';
                } else {
                    c = '&';
                }
                effectiveUri = callback + c + "timestamp=" + timestamp;
            }

            try {
                URI uri = new URI(effectiveUri).parseServerAuthority();
                if (uri.getHost() == null) {
                    throw new BadRequestException("Invalid host for callback " + effectiveUri);
                }
                parsedCallbacks.add(uri);
            } catch (URISyntaxException e) {
                throw new BadRequestException("Bad callback: " + callback);
            }
        }

        return parsedCallbacks;
    }

    public List<URI> faceCallbacks() {
        return faceCallbacks;
    }

    public OcrProxy proxy() {
        return proxy;
    }

    public ProxySession proxySession() {
        return proxySession;
    }

    public String stid() {
        return stid;
    }

    public Integer imageWidth() {
        return imageWidth;
    }

    public Integer imageHeight() {
        return imageHeight;
    }

    public boolean updateOnEmpty() {
        return updateOnEmpty;
    }

    /**
     * Stid for handling in ocr/cv
     * @return
     */
    public String unistorageKey() {
        return unistorageKey;
    }

    public void adjustSrwHeaders(
        final BasicAsyncRequestProducerGenerator generator)
    {
        if (avaKey) {
            generator.addHeader(DiskParams.AVATAR_X_SRW_KEY_TYPE);
            generator.addHeader(DiskParams.AVATAR_X_SRW_NAMESPACE);
        } else {
            generator.addHeader(DiskParams.DISK_X_SRW_KEY_TYPE);
            generator.addHeader(DiskParams.DISK_X_SRW_NAMESPACE);
        }

        generator.addHeader(YandexHeaders.X_SRW_KEY, processStid);

        generator.addHeader(
            YandexHeaders.X_YA_SERVICE_TICKET,
            proxy.apeTvm2Ticket());
        generator.addHeader(
            YandexHeaders.X_SRW_SERVICE_TICKET,
            proxy.unistorageTvm2Ticket());
    }

    public long prefix() {
        return prefix;
    }

    public String id() {
        return id;
    }

    public Long timestamp() {
        return timestamp;
    }

    public Header zooShardId() {
        return zooShardId;
    }

    public boolean isPreviewAvaStid() {
        return avaKey;
    }

    public long lag() {
        if (timestamp == null) {
            return 0L;
        } else {
            long now =
                TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
            return now - timestamp;
        }
    }

    public boolean hasCallbacks() {
        return callbacks.size() + faceCallbacks.size() > 0;
    }

    public List<URI> callbacks() {
        return callbacks;
    }

    // CSOFF: ParameterNumber
    public FutureCallback<Object> spawnCallbacks(
        final AsyncClient client,
        final String body,
        final Charset requestCharset,
        final String queueName,
        final boolean queueIdInHash,
        final String hashPrefixStr,
        final List<URI> callbacks,
        final FutureCallback<Object> callback)
    {
        if (callbacks.isEmpty()) {
            return callback;
        }

        // create hash
        StringBuilder hash = new StringBuilder(hashPrefixStr);
        hash.append('f');

        String queueNamePrefix = HexStrings.LOWER.toString(
            queueName.getBytes(StandardCharsets.UTF_8));
        hash.append(queueNamePrefix);
        hash.append('f');

        String prefixHash = Long.toHexString(prefix);
        for (int i = prefixHash.length(); i < PREFIX_PADDING; ++i) {
            hash.append('0');
        }

        if (queueIdInHash) {
            String zooQueueIdHex = Long.toHexString(cgiZooQueueId);
            for (int i = zooQueueIdHex.length(); i < QUEUE_ID_PADDING; ++i) {
                hash.append('0');
            }
            hash.append(zooQueueIdHex);
            hash.append('f');
        }

        hash.append(prefixHash);
        hash.append('f');
        String stidHash = HexStrings.LOWER.toString(
            stid.getBytes(StandardCharsets.UTF_8));
        hash.append(stidHash);
        hash.append('f');

        String hashPrefix = new String(hash);

        MultiFutureCallback<Object> multiCallback =
            new MultiFutureCallback<>(callback);
        List<Header> headers = new ArrayList<>(2);
        headers.add(new BasicHeader(YandexHeaders.SERVICE, queueName));
        headers.add(zooShardId);
        headers.add(new BasicHeader(YandexHeaders.ZOO_HASH, hashPrefix));
        headers.add(CHECK_DUPLICATE);

        ContentType contentType =
            ContentType.APPLICATION_JSON.withCharset(requestCharset);
        AsyncClient callbacksClient = client.adjust(proxySession.context());
        Supplier<? extends HttpClientContext> contextGenerator =
            proxySession.listener().createContextGeneratorFor(callbacksClient);
        for (URI callbackURI: callbacks) {
            callbacksClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncPostURIRequestProducerSupplier(
                        callbackURI,
                        body,
                        contentType),
                    headers),
                EmptyAsyncConsumerFactory.OK,
                contextGenerator,
                multiCallback.newCallback());
        }
        FutureCallback<Object> nextCallback = multiCallback.newCallback();
        multiCallback.done();
        return nextCallback;
    }
    // CSON: ParameterNumber
}

