package ru.yandex.search.disk.proxy;

import java.util.HashSet;
import java.util.Set;

import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.erratum.ErratumConfig;
import ru.yandex.erratum.ErratumConfigBuilder;
import ru.yandex.geocoder.GeocoderConfig;
import ru.yandex.geocoder.GeocoderConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.URIConfig;
import ru.yandex.http.config.URIConfigBuilder;
import ru.yandex.jniwrapper.JniWrapperConfig;
import ru.yandex.jniwrapper.JniWrapperConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.search.proxy.universal.AbstractUniversalSearchProxyConfigBuilder;

public abstract class AbstractProxyConfigBuilder
    <T extends AbstractProxyConfigBuilder<T>>
    extends AbstractUniversalSearchProxyConfigBuilder<T>
    implements ProxyConfig
{
    private ErratumConfigBuilder misspellConfig;
    private GeocoderConfigBuilder geoSearchConfig;
    private HttpHostConfigBuilder userSplitConfig;
    private HttpHostConfigBuilder faceIndexerConfig;
    private String imagesFilter;
    private String searchPostFilter;
    private String diskService;
    private String photosliceService;
    private String ipddService;
    private Set<String> producerServices;
    private long fatUserDocs;
    private HttpTargetConfigBuilder photosliceClientConfig;
    private JniWrapperConfigBuilder dssmConfig;
    private long dssmThreshold;
    private URIConfigBuilder djfsConfig;
    private Tvm2ClientConfigBuilder tvm2ClientConfig;
    private Long failoverSearchDelay;
    private String djfsTvmClientId;
    private String geoTvmClientId;

    protected AbstractProxyConfigBuilder(final ProxyConfig config) {
        super(config);
        misspellConfig(config.misspellConfig());
        geoSearchConfig(config.geoSearchConfig());
        imagesFilter(config.imagesFilter());
        searchPostFilter(config.searchPostFilter());
        diskService(config.diskService());
        photosliceService(config.photosliceService());
        ipddService(config.ipddService());
        producerServices(config.producerServices());
        fatUserDocs(config.fatUserDocs());
        photosliceClientConfig(config.photosliceClientConfig());
        dssmConfig(config.dssmConfig());
        dssmThreshold(config.dssmThreshold());
        djfsConfig(config.djfsConfig());
        tvm2ClientConfig(config.tvm2ClientConfig());
        failoverSearchDelay(config.failoverSearchDelay());
        djfsTvmClientId(config.djfsTvmClientId());
        geoTvmClientId(config.geoTvmClientId());
        userSplitConfig(config.userSplitConfig());
    }

    protected AbstractProxyConfigBuilder(
        final IniConfig config,
        final ProxyConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        IniConfig misspellConfig = config.sectionOrNull("misspell");
        if (misspellConfig == null) {
            misspellConfig(defaults.misspellConfig());
        } else if (defaults.misspellConfig() == null) {
            this.misspellConfig =
                new ErratumConfigBuilder(misspellConfig);
        } else {
            this.misspellConfig =
                new ErratumConfigBuilder(
                    misspellConfig,
                    defaults.misspellConfig());
        }

        IniConfig userSplitSection = config.sectionOrNull("user-split");
        if (userSplitSection != null) {
            userSplitConfig =
                new HttpHostConfigBuilder(
                    userSplitSection,
                    defaults.userSplitConfig());
        } else {
            userSplitConfig = null;
        }

        IniConfig geoSearchConfig = config.sectionOrNull("geo");
        if (geoSearchConfig == null) {
            geoSearchConfig(defaults.geoSearchConfig());
        } else if (defaults.geoSearchConfig() == null) {
            this.geoSearchConfig =
                new GeocoderConfigBuilder(geoSearchConfig);
        } else {
            this.geoSearchConfig =
                new GeocoderConfigBuilder(
                    geoSearchConfig,
                    defaults.geoSearchConfig());
        }

        IniConfig faceIndexerIniConfig = config.sectionOrNull("face-indexer");
        if (faceIndexerIniConfig == null) {
            faceIndexerConfig = null;
        } else {
            faceIndexerConfig =
                new HttpHostConfigBuilder(
                    faceIndexerIniConfig,
                    defaults.faceIndexerConfig());
        }
        IniConfig proxy = config.section("proxy");

        imagesFilter = proxy.get(
            "images-filter",
            defaults.imagesFilter(),
            NonEmptyValidator.INSTANCE);
        searchPostFilter =
            proxy.getString("search-post-filter", defaults.searchPostFilter());
        diskService = proxy.get(
            "disk-service",
            defaults.diskService(),
            NonEmptyValidator.INSTANCE);
        photosliceService = proxy.get(
            "photoslice-service",
            defaults.photosliceService(),
            NonEmptyValidator.INSTANCE);
        ipddService = proxy.get(
            "ipdd-service",
            defaults.ipddService(),
            NonEmptyValidator.INSTANCE);
        producerServices = proxy.get(
            "producer-services",
            defaults.producerServices(),
            new CollectionParser<>(NonEmptyValidator.TRIMMED, HashSet::new));
        fatUserDocs = proxy.get(
            "fat-user-docs",
            defaults.fatUserDocs(),
            NonNegativeLongValidator.INSTANCE);

        photosliceClientConfig = new HttpTargetConfigBuilder(
            config.section("photoslice-client"),
            defaults.photosliceClientConfig());

        IniConfig dssmSection = config.section("dssm");
        dssmConfig = new JniWrapperConfigBuilder(
            dssmSection,
            defaults.dssmConfig());
        dssmThreshold = dssmSection.get(
            "threshold",
            defaults.dssmThreshold(),
            Long::valueOf);

        djfsConfig = new URIConfigBuilder(
            config.section("djfs"),
            defaults.djfsConfig());
        tvm2ClientConfig = new Tvm2ClientConfigBuilder(
            config.section("tvm2"),
            defaults.tvm2ClientConfig());
        failoverSearchDelay =
            config.getLongDuration(
                "failover-search-delay",
                defaults.failoverSearchDelay());
        djfsTvmClientId = config.get(
            "djfs.tvm-client-id",
            defaults.djfsTvmClientId(),
            NonEmptyValidator.INSTANCE
        );
        geoTvmClientId = config.get(
            "geo.tvm-client-id",
            defaults.geoTvmClientId(),
            NonEmptyValidator.INSTANCE
        );
    }

    @Override
    public ErratumConfigBuilder misspellConfig() {
        return misspellConfig;
    }

    public T misspellConfig(final ErratumConfig misspellConfig) {
        if (misspellConfig == null) {
            this.misspellConfig = null;
        } else {
            this.misspellConfig = new ErratumConfigBuilder(misspellConfig);
        }
        return self();
    }

    @Override
    public GeocoderConfigBuilder geoSearchConfig() {
        return geoSearchConfig;
    }

    public T geoSearchConfig(final GeocoderConfig geoSearchConfig) {
        if (geoSearchConfig == null) {
            this.geoSearchConfig = null;
        } else {
            this.geoSearchConfig = new GeocoderConfigBuilder(geoSearchConfig);
        }
        return self();
    }

    @Override
    public String imagesFilter() {
        return imagesFilter;
    }

    public T imagesFilter(final String imagesFilter) {
        this.imagesFilter = imagesFilter;
        return self();
    }

    @Override
    public String searchPostFilter() {
        return searchPostFilter;
    }

    public T searchPostFilter(final String searchPostFilter) {
        this.searchPostFilter = searchPostFilter;
        return self();
    }

    @Override
    public String diskService() {
        return diskService;
    }

    public T diskService(final String diskService) {
        this.diskService = diskService;
        return self();
    }

    @Override
    public String photosliceService() {
        return photosliceService;
    }

    public T photosliceService(final String photosliceService) {
        this.photosliceService = photosliceService;
        return self();
    }

    @Override
    public String ipddService() {
        return ipddService;
    }

    public T ipddService(final String ipddService) {
        this.ipddService = ipddService;
        return self();
    }

    @Override
    public Set<String> producerServices() {
        return producerServices;
    }

    public T producerServices(final Set<String> producerServices) {
        this.producerServices = producerServices;
        return self();
    }

    @Override
    public long fatUserDocs() {
        return fatUserDocs;
    }

    public T fatUserDocs(final long fatUserDocs) {
        this.fatUserDocs = fatUserDocs;
        return self();
    }

    @Override
    public HttpTargetConfigBuilder photosliceClientConfig() {
        return photosliceClientConfig;
    }

    public T photosliceClientConfig(
        final HttpTargetConfig photosliceClientConfig)
    {
        this.photosliceClientConfig =
            new HttpTargetConfigBuilder(photosliceClientConfig);
        return self();
    }

    @Override
    public JniWrapperConfigBuilder dssmConfig() {
        return dssmConfig;
    }

    public T dssmConfig(final JniWrapperConfig dssmConfig) {
        this.dssmConfig = new JniWrapperConfigBuilder(dssmConfig);
        return self();
    }

    @Override
    public long dssmThreshold() {
        return dssmThreshold;
    }

    public T dssmThreshold(final long dssmThreshold) {
        this.dssmThreshold = dssmThreshold;
        return self();
    }

    @Override
    public URIConfigBuilder djfsConfig() {
        return djfsConfig;
    }

    public T djfsConfig(final URIConfig djfsConfig) {
        this.djfsConfig = new URIConfigBuilder(djfsConfig);
        return self();
    }

    @Override
    public Tvm2ClientConfigBuilder tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public T tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig) {
        this.tvm2ClientConfig = new Tvm2ClientConfigBuilder(tvm2ClientConfig);
        return self();
    }

    @Override
    public Long failoverSearchDelay() {
        return failoverSearchDelay;
    }

    public T failoverSearchDelay(final Long failoverSearchDelay) {
        this.failoverSearchDelay = failoverSearchDelay;
        return self();
    }

    @Override
    public String djfsTvmClientId() {
        return djfsTvmClientId;
    }

    public T djfsTvmClientId(final String djfsTvmClientId) {
        this.djfsTvmClientId = djfsTvmClientId;
        return self();
    }

    @Override
    public String geoTvmClientId() {
        return geoTvmClientId;
    }

    public T geoTvmClientId(final String geoTvmClientId) {
        this.geoTvmClientId = geoTvmClientId;
        return self();
    }

    @Override
    public HttpHostConfigBuilder userSplitConfig() {
        return userSplitConfig;
    }

    public T userSplitConfig(final HttpHostConfig config) {
        if (config != null) {
            this.userSplitConfig = new HttpHostConfigBuilder(config);
        } else {
            this.userSplitConfig = null;
        }

        return self();
    }

    @Override
    public HttpHostConfigBuilder faceIndexerConfig() {
        return faceIndexerConfig;
    }

    public T faceIndexerConfig(final HttpHostConfig faceIndexerConfig) {
        this.faceIndexerConfig = new HttpHostConfigBuilder(faceIndexerConfig);
        return self();
    }
}

