package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.util.string.HexStrings;

public class DiskResultPrinter extends AbstractDiskResultPrinter {
    private static final String ATTR = "attr";
    private static final String DOCS = "docs";
    private static final String HOW = "how";
    private static final String ONE = "1";
    private static final String RELEVANCE = "relevance";
    private static final String SCOPE = "scope";
    private static final String PREFIX = "prefix";
    private static final String ID = "id";

    private final IdentityHashMap<String, long[]> scopes =
        new IdentityHashMap<>();
    private final Proxy proxy;
    private final DiskRequestParams requestParams;

    // CSOFF: ParameterNumber
    public DiskResultPrinter(
        final Proxy proxy,
        final ProxySession session,
        final DiskRequestParams requestParams,
        final SearchInfo searchInfo)
        throws BadRequestException
    {
        super(session, searchInfo);
        this.proxy = proxy;
        this.requestParams = requestParams;
    }
    // CSOFF: ParameterNumber

    @Override
    public void completed(final SearchResult searchResult) {
        super.completed(searchResult);
        HttpHost producerHost = proxy.producerHost();
        String normalized =
            SearchRequestText.normalizeSuggest(searchInfo.request());
        if (producerHost == null || normalized.isEmpty()) {
            return;
        }
        StringBuilder sb = new StringBuilder("history_suggest_");
        HexStrings.LOWER.toStringBuilder(
            sb,
            normalized.toLowerCase(Locale.ROOT).replace('ё', 'е')
                .getBytes(StandardCharsets.UTF_8));
        String id = new String(sb);
        long hitsCount = searchResult.hitsCount();
        session.logger().info(
            "Accounting suggest \"" + normalized + '"' + '(' + id
            + ") with hits count: " + hitsCount);
        try {
            QueryConstructor query;
            if (hitsCount == 0L) {
                query = new QueryConstructor("/delete?history-suggest");
                proxy.historySuggestIndexed();
            } else {
                query = new QueryConstructor("/modify?history-suggest");
                proxy.historySuggestRemoved();
            }

            query.append(PREFIX, requestParams.user().prefix().toString());
            query.append("service", requestParams.user().service());
            query.append(ID, id);

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                writer.startObject();
                writer.key(PREFIX);
                requestParams.user().prefix().writeValue(writer);
                writer.key(DOCS);
                writer.startArray();
                writer.startObject();
                writer.key(ID);
                writer.value(id);
                writer.key("resource_id");
                writer.value(id);
                if (hitsCount != 0L) {
                    writer.key("history_suggest_request");
                    writer.value(normalized);
                    writer.key("history_suggest_timestamp");
                    writer.value(System.currentTimeMillis());
                }
                writer.endObject();
                writer.endArray();
                writer.endObject();
            }
            String uri = query.toString();
            proxy.producerClient().execute(
                producerHost,
                new BasicAsyncRequestProducerGenerator(
                    uri,
                    sbw.toString(),
                    proxy.indexerContentType()),
                EmptyAsyncConsumerFactory.ANY_GOOD,
                new UpstreamStaterFutureCallback<>(
                    new HistorySuggestFutureCallback(session.logger(), uri),
                    proxy.historySuggestIndexationStater()));
        } catch (HttpException | IOException e) {
            session.logger().log(
                Level.WARNING,
                "Failed to index history suggest",
                e);
        }
    }

    private void writeSearchInfo(final JsonWriter writer) throws IOException {
        writer.key("request");
        writer.value(searchInfo.request());
        String suggest = searchInfo.suggest();
        if (suggest != null) {
            writer.key("suggest");
            writer.value(suggest);
        }
        String rule = searchInfo.erratumRule();
        if (rule != null) {
            writer.key("rule");
            writer.value(rule);
        }
    }

    private void writePageInfo(final JsonWriter writer) throws IOException {
        writer.key("sortBy");

        writer.startObject();
        writer.key(HOW);
        writer.value(session.params().getString(HOW, ""));
        writer.key("order");
        if (requestParams.asc()) {
            writer.value("ascending");
        } else {
            writer.value("descending");
        }
        writer.key("priority");
        writer.value("no");
        writer.endObject();

        writer.key("groupings");
        writer.startArray();
        writer.startObject();
        writer.key(ATTR);
        writer.value("");
        writer.key("categ");
        writer.value("");
        writer.key(DOCS);
        writer.value(ONE);
        writer.key("groups-on-page");
        writer.value(Integer.toString(requestParams.length()));
        writer.key("mode");
        writer.value("flat");
        writer.endObject();
        writer.endArray();
    }

    private void writeDoc(
        final JsonWriter writer,
        final Map<String, String> attrs)
        throws IOException
    {
        String id = attrs.get(ID);
        writer.startObject();
        writer.key("doccount");
        writer.value(ONE);
        writer.key(RELEVANCE);
        writer.value(id);
        writer.key("documents");

        writer.startArray();
        writer.startObject();
        writer.key("docId");
        writer.value(id);
        writer.key("url");
        writer.value("");
        writer.key(RELEVANCE);
        writer.value(id);

        writer.key("properties");
        writer.startObject();
        for (String field: requestParams.fields()) {
            String value = attrs.get(field);
            if (value != null) {
                writer.key(field);
                writer.value(value);
            }
        }
        String scope = attrs.get(SCOPE);
        ++scopes.computeIfAbsent(scope, x -> new long[1])[0];
        writer.key(SCOPE);
        writer.value(scope);
        writer.endObject();

        writer.endObject();
        writer.endArray();

        writer.endObject();
    }

    private static void writeFound(
        final JsonWriter writer,
        final String all,
        final String phrase)
        throws IOException
    {
        writer.key("found");
        writer.startObject();
        writer.key("all");
        writer.value(all);
        writer.key("phrase");
        writer.value(phrase);
        writer.key("strict");
        writer.value(phrase);
        writer.endObject();
    }

    @Override
    protected void writeSerp(
        final JsonWriter writer,
        final SearchResult searchResult)
        throws IOException
    {
        writer.startObject();
        writeSearchInfo(writer);
        writePageInfo(writer);

        writer.key("response");
        writer.startObject();

        List<SearchDocument> docs = searchResult.hitsArray();
        int size = docs.size();
        String count =
            Integer.toString(Math.max(0, size - requestParams.offset()));
        writeFound(writer, count, count);

        writer.key("results");
        writer.startArray();
        writer.startObject();

        writer.key(ATTR);
        writer.value("");
        writer.key(DOCS);
        writer.value(ONE);

        writeFound(writer, count, "0");

        writer.key("groups");
        writer.startArray();
        int max = Math.min(
            requestParams.offset() + requestParams.length(),
            size);
        for (int i = requestParams.offset(); i < max; ++i) {
            writeDoc(writer, docs.get(i).attrs());
        }
        writer.endArray();

        writer.endObject();
        writer.endArray();

        writer.endObject();
        writer.endObject();

        proxy.searchCompleted(scopes, size == 0);
    }

    private static class HistorySuggestFutureCallback
        implements FutureCallback<Void>
    {
        private final Logger logger;
        private final String uri;

        HistorySuggestFutureCallback(
            final Logger logger,
            final String uri)
        {
            this.logger = logger;
            this.uri = uri;
        }

        @Override
        public void cancelled() {
            logger.warning("Suggest indexing request cancelled: " + uri);
        }

        @Override
        public void completed(final Void response) {
            logger.info("Suggest indexing request completed: " + uri);
        }

        @Override
        public void failed(final Exception e) {
            logger.log(
                Level.WARNING,
                "Suggest indexing request failed: " + uri,
                e);
        }
    }
}

