package ru.yandex.search.disk.proxy;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.search.prefix.LongPrefix;

public class DropIndexHandler implements ProxyRequestHandler {
    private final Proxy proxy;

    public DropIndexHandler(final Proxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        Context context = new Context(session, proxy);
        MultiFutureCallback<Void> callback =
            new MultiFutureCallback<>(new SearchCallback(context));
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(
                "/search?text=type:*&length=0&prefix=" + context.uid());
        AsyncClient client = proxy.searchClient().adjust(session.context());
        Supplier<? extends HttpClientContext> contextSupplier =
            session.listener().createContextGeneratorFor(client);
        for (HttpHost searchHost: context.searchHosts()) {
            client.execute(
                searchHost,
                producerGenerator,
                EmptyAsyncConsumerFactory.OK,
                contextSupplier,
                callback.newCallback());
        }
        callback.done();
    }

    @Override
    public String toString() {
        return "Index dropping handler";
    }

    private static class Context {
        private final ProxySession session;
        private final long uid;
        private final List<HttpHost> searchHosts;
        private final List<HttpHost> indexerHosts;
        private final AsyncClient indexerClient;
        private final Supplier<? extends HttpClientContext> contextSupplier;

        Context(final ProxySession session, final Proxy proxy)
            throws BadRequestException
        {
            this.session = session;
            uid = session.params().get(
                "uid",
                NonNegativeLongValidator.INSTANCE);
            SearchMapShard shard = proxy.searchMap().hosts(
                new User(proxy.diskService(), new LongPrefix(uid)));
            searchHosts = new ArrayList<>(shard.size());
            indexerHosts = new ArrayList<>(shard.size());
            for (SearchMapHost searchMapHost: shard) {
                HttpHost searchHost = searchMapHost.searchHost();
                if (searchHost == null) {
                    throw new BadRequestException(
                        "No search host for " + searchMapHost);
                }
                searchHosts.add(searchHost);

                HttpHost indexerHost = searchMapHost.indexerHost();
                if (indexerHost == null) {
                    throw new BadRequestException(
                        "No indexer host for " + searchMapHost);
                }
                indexerHosts.add(indexerHost);
            }
            indexerClient = proxy.indexerClient().adjust(session.context());
            contextSupplier =
                session.listener().createContextGeneratorFor(indexerClient);
        }

        public ProxySession session() {
            return session;
        }

        public long uid() {
            return uid;
        }

        public List<HttpHost> searchHosts() {
            return searchHosts;
        }

        public List<HttpHost> indexerHosts() {
            return indexerHosts;
        }

        public AsyncClient indexerClient() {
            return indexerClient;
        }

        public Supplier<? extends HttpClientContext> contextSupplier() {
            return contextSupplier;
        }
    }

    private static class SearchCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final Context context;

        SearchCallback(final Context context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Object result) {
            MultiFutureCallback<Void> callback =
                new MultiFutureCallback<>(new DeleteCallback(context));
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    "/delete?text=type:*&yes_i_want_this=ugu&prefix="
                    + context.uid());
            for (HttpHost indexerHost: context.indexerHosts()) {
                context.indexerClient().execute(
                    indexerHost,
                    producerGenerator,
                    EmptyAsyncConsumerFactory.OK,
                    context.contextSupplier(),
                    callback.newCallback());
            }
            callback.done();
        }
    }

    private static class DeleteCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final Context context;

        DeleteCallback(final Context context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Object result) {
            final long shards = 100L;
            MultiFutureCallback<Void> callback =
                new MultiFutureCallback<>(new FlushCallback(context));
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    "/flush?wait=true&shards=" + (context.uid() % shards));
            for (HttpHost indexerHost: context.indexerHosts()) {
                context.indexerClient().execute(
                    indexerHost,
                    producerGenerator,
                    EmptyAsyncConsumerFactory.OK,
                    context.contextSupplier(),
                    callback.newCallback());
            }
            callback.done();
        }
    }

    private static class FlushCallback
        extends AbstractProxySessionCallback<Object>
    {
        FlushCallback(final Context context) {
            super(context.session());
        }

        @Override
        public void completed(final Object result) {
            session.response(HttpStatus.SC_OK);
        }
    }
}

