package ru.yandex.search.disk.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.collection.LongPair;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.SequentialCompleteMultiFutureCallback;
import ru.yandex.http.util.TimingFutureCallback;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.disk.proxy.rules.NamedRule;
import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class ExperimentMultiSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private final NamedRule[] rules;
    private final Consumer<? super MultiSearchStat> stater;

    public ExperimentMultiSearchRule(
        final List<NamedRule> rules,
        final Consumer<? super MultiSearchStat> stater)
    {
        this.rules = rules.toArray(new NamedRule[rules.size()]);
        this.stater = stater;
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
            request)
        throws HttpException
    {
        int maxDocs = request.requestParams().offset()
            + request.requestParams().length();
        SequentialCompleteMultiFutureCallback<LongPair<Map.Entry<String, SearchResult>>> callback =
            new SequentialCompleteMultiFutureCallback<>(
                new Callback(request, stater, rules.length, maxDocs),
                request.session().logger(),
                rules.length);

        for (int i = 0; i < rules.length; ++i) {
            NamedRule rule = rules[i];
            rule.execute(
                request.withCallback(
                    new PayloadFutureCallback<>(
                        rule.name(),
                        new TimingFutureCallback<>(
                            callback.next()))));
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<
        List<LongPair<Map.Entry<String, SearchResult>>>,
        SearchResult>
    {
        private final long start = System.currentTimeMillis();
        private final int maxDocs;
        private final Consumer<? super MultiSearchStat> stater;
        private final Map<String, Long> timings;
        private final PrefixedLogger logger;
        private long hitsCount = 0L;
        private int rulesCount;
        private volatile boolean done = false;

        Callback(
            final SearchRequest<SearchResult, DiskRequestParams, SearchInfo> request,
            final Consumer<? super MultiSearchStat> stater,
            final int size,
            final int maxDocs)
        {
            super(request.callback());
            this.stater = stater;
            this.logger = request.session().logger();
            this.maxDocs = maxDocs;
            this.rulesCount = size;
            this.timings = new HashMap<>(size);
        }

        @Override
        public void completed(
            final List<LongPair<Map.Entry<String, SearchResult>>> results)
        {
            boolean done = false;
            List<SearchDocument> docs = null;

            try {
                synchronized (this) {
                    if (this.done) {
                        return;
                    }

                    for (LongPair<Map.Entry<String, SearchResult>> result : results) {
                        hitsCount += result.second().getValue().hitsCount();
                    }

                    Set<String> ids = new HashSet<>(maxDocs << 1);
                    docs = new ArrayList<>(maxDocs);
                    for (LongPair<Map.Entry<String, SearchResult>> result : results) {
                        String ruleName = result.second().getKey();
                        timings.put(ruleName, result.first());
                        for (SearchDocument document: result.second().getValue().hitsArray()) {
                            if (ids.add(document.attrs().get("id"))) {
                                docs.add(document);
                                if (docs.size() >= maxDocs) {
                                    done = true;
                                    this.done = true;
                                    break;
                                }
                            }
                        }
                        if (docs.size() >= maxDocs) {
                            done = true;
                            this.done = true;
                            break;
                        }
                    }

                    if (rulesCount <= results.size()) {
                        done = true;
                        this.done = true;
                    }
                }

                if (done) {
                    long totalTime = System.currentTimeMillis() - start;
                    stater.accept(new MultiSearchStat(totalTime, timings));
                    callback.completed(new BasicSearchResult(docs, hitsCount));
                }
            } catch (Exception e) {
                logger.log(Level.WARNING, "Experimental Rule Exception", e);
                callback.failed(e);
            }

        }

        @Override
        public void cancelled() {
            if (!done) {
                super.cancelled();
            }

        }

        @Override
        public void failed(Exception e) {
            if (!done) {
                super.failed(e);
            }
        }
    }
}

