package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.disk.search.DiskField;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class FolderResourcesHandler implements ProxyRequestHandler {
    private static final Set<String> DEFAULT_GET = Collections.singleton("resource_id");
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);

    private static final long FAILOVER_DELAY = 300L;

    private final Proxy proxy;

    public FolderResourcesHandler(final Proxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Context context = new Context(session);

        StringBuilder sb = new StringBuilder();
        sb.append(DiskField.PARENT_FID.fieldName());
        sb.append(':');
        sb.append(context.fid());
        sb.append(" AND ");
        sb.append("type:file");
        sb.append(" AND NOT ");
        sb.append("mediatype:(");
        sb.append(StringUtils.join(context.medias(), ' '));
        sb.append(")");
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());
        qc.append("service", "disk_queue");
        qc.append("text", sb.toString());
        qc.append("length", 1);
        qc.append("get", StringUtils.join(context.get(), ','));

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new Callback(session, context));
    }

    private static final class Callback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Context context;

        public Callback(
            final ProxySession session,
            final Context context)
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType().create(sbw)) {
                JsonMap result = resultObj.asMap();
                long total = result.getLong("hitsCount");
                JsonList results = result.getList("hitsArray");
                writer.startObject();
                writer.key("count");
                writer.value(total);
                writer.key("result");
                writer.startArray();
                for (int i = 0; i < Math.min(context.length, results.size()); i++) {
                    writer.value(results.get(i));
                }
                writer.endArray();
                writer.endObject();
            } catch (IOException | JsonException e) {
                failed(e);
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }

    private final class Context implements UniversalSearchProxyRequestContext {
        private final Set<String> medias;
        private final Set<String> get;
        private final String fid;
        private final Long uid;
        private final User user;
        private final int length;
        private final boolean allowLaggingHosts;

        private final ProxySession session;
        private final AsyncClient client;
        private final JsonType jsonType;

        public Context(final ProxySession session) throws BadRequestException {
            this.session = session;
            this.uid = session.params().getLong("uid");
            this.length = session.params().getInt("length", 1);
            this.fid = session.params().getString("fid");
            this.medias = session.params().get("exclude-media", SET_PARSER);
            this.get = session.params().get("get", DEFAULT_GET, SET_PARSER);
            this.user = new User("disk_queue", new LongPrefix(uid));
            allowLaggingHosts =
                session.params().getBoolean("allow-lagging-hosts", false);
            client = proxy.searchClient().adjust(session.context());
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        public Set<String> get() {
            return get;
        }

        public int length() {
            return length;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public Set<String> medias() {
            return medias;
        }

        public String fid() {
            return fid;
        }

        public Long uid() {
            return uid;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE: 0L;
        }
    }
}
