package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.disk.search.DiskParams;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterFactory;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class FolderSizeHandler extends AbstractUidProxyHandler {
    private static final int FAILOVERDELAY = 5000;
    private static final int TIMEOUT = 60000;

    public FolderSizeHandler(final Proxy proxy) {
        super(proxy, proxy.diskService());
    }

    // CSOFF: MultipleStringLiterals
    @Override
    public void handle(
        final ProxySession session,
        final User user,
        final List<HttpHost> hosts)
        throws HttpException
    {
        AsyncClient client = proxy.searchClient().adjust(session.context());

        StringBuilder text;
        QueryConstructor query;
        if (session.params().getBoolean(DiskParams.FAST_MOVED, false)) {
            Set<String> resIds = session.params().getAll(
                "resource_id",
                NonEmptyValidator.INSTANCE,
                new LinkedHashSet<>());

            query = new QueryConstructor(
                "/search-folder-size?"
                    + "&get=total_size,total_count,resource_id"
                    + "&dp=tree_calc(parent_fid,fid,"
                    + "sum_count(size,type,file+total_size,total_count))"
                    + "&dp-tree-max-height=5000000&fast-moved");
            text = new StringBuilder("resource_id:(");
            int i = 0;
            for (String resId: resIds) {
                if (i != 0) {
                    text.append(" OR ");
                }
                i++;
                text.append(
                    SearchRequestText.fullEscape(
                        resId,
                        false));
            }

            text.append(')');
            query.append("prefix", user.prefix().toString());
            query.append("text", text.toString());
            client.execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(query.sb().toString()),
                System.currentTimeMillis() + TIMEOUT,
                FAILOVERDELAY,
                SearchResultConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new FastMovedCallback(
                    session,
                    resIds,
                    JsonTypeExtractor.NORMAL.extract(session.params())));
            return;
        }
        List<String> keys = session.params().getAll(
            "key",
            NonEmptyValidator.INSTANCE,
            new ArrayList<String>());
        MultiFutureCallback<SearchResult> callback =
            new MultiFutureCallback<>(
                new Callback(
                    session,
                    keys,
                    JsonTypeExtractor.NORMAL.extract(session.params())));

        query = new QueryConstructor(
            "/search-folder-size?get=total_size&group=type"
            + "&aggregate=sum(size)+total_size&merge_func=count");
        query.append("prefix", user.prefix().toString());
        int queryPrefixLength = query.sb().length();
        text = new StringBuilder("type:file");
        int textPrefixLength = text.length();
        for (String key: keys) {
            text.setLength(textPrefixLength);
            text.append(" AND key:");
            text.append(
                SearchRequestText.fullEscape(key, false)
                    .replace('Ё', 'Е')
                    .replace('ё', 'е'));
            text.append('*');
            query.sb().setLength(queryPrefixLength);
            query.append("text", text.toString());
            client.execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(query.sb().toString()),
                SearchResultConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                callback.newCallback());
        }
        callback.done();
    }

    @Override
    public String toString() {
        return "Calculates total size of files in folder and all subfolders";
    }

    private static final class FastMovedCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final Set<String> resIds;
        private final JsonWriterFactory writerFactory;

        private FastMovedCallback(
            final ProxySession session,
            final Set<String> resIds, final JsonWriterFactory writerFactory)
        {
            super(session);
            this.resIds = resIds;
            this.writerFactory = writerFactory;
        }

        @Override
        public void completed(final SearchResult result) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            int i = 0;
            try (JsonWriter writer = writerFactory.create(sbw)) {
                writer.startObject();
                for (SearchDocument doc: result.hitsArray()) {
                    String resId = doc.attrs().get("resource_id");
                    String totalSizeStr = doc.attrs().get("total_size");
                    String totalCountStr = doc.attrs().get("total_count");
                    if (!resIds.contains(resId)) {
                        session.logger().warning("Bad res id " + resId);
                        continue;
                    }

                    long totalSize = 0;
                    long totalCount = 0;

                    if (totalSizeStr != null) {
                        totalSize = ValueUtils.asLong(totalSizeStr);
                    }

                    if (totalCountStr != null) {
                        totalCount = ValueUtils.asLong(totalCountStr);
                    }

                    writer.key(resId);
                    writer.startObject();
                    writer.key("size");
                    writer.value(totalSize);
                    writer.key("count");
                    writer.value(totalCount);
                    writer.endObject();
                }
                writer.endObject();
            } catch (IOException | JsonUnexpectedTokenException e) {
                failed(
                    new BadGatewayException(
                        "Failed to parse result #" + i + ':' + ' ' + result,
                        e));
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON.withCharset(
                        session.acceptedCharset())));
        }
    }

    private static class Callback
        extends AbstractProxySessionCallback<List<SearchResult>>
    {
        private final List<String> keys;
        private final JsonWriterFactory writerFactory;

        Callback(
            final ProxySession session,
            final List<String> keys,
            final JsonWriterFactory writerFactory)
        {
            super(session);
            this.keys = keys;
            this.writerFactory = writerFactory;
        }

        @Override
        public void completed(final List<SearchResult> results) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            int i = 0;
            try (JsonWriter writer = writerFactory.create(sbw)) {
                writer.startObject();
                for (; i < keys.size(); ++i) {
                    List<SearchDocument> hits = results.get(i).hitsArray();
                    int size = hits.size();
                    if (size == 1) {
                        Map<String, String> attrs = hits.get(0).attrs();
                        writer.key(keys.get(i));
                        writer.startObject();
                        writer.key("size");
                        writer.value(
                            ValueUtils.asLong(attrs.get("total_size")));
                        writer.key("count");
                        writer.value(
                            ValueUtils.asInt(attrs.get("merged_docs_count"), 0)
                            + 1);
                        writer.endObject();
                    } else if (size > 1) {
                        failed(
                            new BadGatewayException(
                                "Too many hits count in result #" + i
                                + ':' + ' ' + results));
                        return;
                    }
                }
                writer.endObject();
            } catch (IOException | JsonUnexpectedTokenException e) {
                failed(
                    new BadGatewayException(
                        "Failed to parse result #" + i + ':' + ' ' + results,
                        e));
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON.withCharset(
                        session.acceptedCharset())));
        }
    }
    // CSON: MultipleStringLiterals
}

