package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class GeoServicesOrgPhotosHandler implements ProxyRequestHandler {
    private final Proxy proxy;
    private final JniWrapper dssm;

    public GeoServicesOrgPhotosHandler(final Proxy proxy, final JniWrapper dssm) {
        this.proxy = proxy;
        this.dssm = dssm;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        GeoServicesOrgPhotosContext context = new GeoServicesOrgPhotosContext(session);

        Set<String> getFields = new LinkedHashSet<>();
        getFields.add("id");
        getFields.add("key");
        getFields.add("stid");
        getFields.add("preview_stid");
        getFields.add("longitude");
        getFields.add("latitude");
        getFields.add("name");
        QueryConstructor qc = new QueryConstructor("/search-geo-org?");
        qc.append("prefix", context.user().prefix().toStringFast());
        qc.append("service", context.user().service());

        String sort = "date";
        boolean asc = false;
        if (context.timerange() != null || context.coordinates() != null || !context.requests().isEmpty()) {
            sort = "date";
        }

        StringBuilder text = new StringBuilder();
        text.append("mediatype:9");
        if (context.coordinates() != null) {
            getFields.add("distance");
            text.append(" AND longitude:* AND longitude:*");
            qc.append("dp", "const(" + context.coordinates().getKey() + " org_lat)");
            qc.append("dp", "const(" + context.coordinates().getValue() + " org_lon)");
            qc.append("dp", "geo_distance(latitude,longitude,org_lat,org_lon distance)");
            if (context.distanceThreshold() != null) {
                qc.append("dp", "filter_cmp(distance,<=," + context.distanceThreshold + ")");
            }
            sort = "distance";
            asc = true;
        }

        if (!context.requests().isEmpty()) {
            text.append(" AND has_i2t:1");
            try {
                int i = 0;
                qc.append("dp", "const(-1 m_one)");
                for (String request: context.requests) {
                    String vector = dssm.apply(request, null).process(null, 0, 0);
                    qc.append("dp", "chex-dot-product(i2t_keyword " + vector + " i2t_" + i + ")");
                    qc.append("dp", "fmul(m_one,i2t_" + i + " i2tt_" + i + ")");
                    i++;
                }
                StringBuilder minFunc = new StringBuilder();
                minFunc.append("min(");
                for (int j = 0; j < i; j++) {
                    minFunc.append("i2tt_");
                    minFunc.append(j);
                    if (j != i -1) {
                        minFunc.append(',');
                    }
                }

                minFunc.append(" i2tscore)");
                qc.append("dp", minFunc.toString());
                qc.append("dp", "fmul(m_one,i2tscore dssmScore)");
                qc.append("dp", "filter_cmp(dssmScore,>=," + context.dssmThreshold + ")");

                getFields.add("dssmScore");
                sort = "dssmScore";
            } catch (JniWrapperException jwe) {
                throw new BadRequestException("Dssm error");
            }
        }

        qc.append("dp", "fallback(etime,created,ctime,modified,mtime date)");
        getFields.add("date");
        if (context.timerange() != null) {
            qc.append("postfilter", "date >= " + context.timerange().getKey());
            qc.append("postfilter", "date <= " + context.timerange().getValue());
        }

        qc.append("text", text.toString());
        qc.append("get", StringUtils.join(getFields, ','));
        qc.append("length", context.length());
        qc.append("offset", context.offset());
        qc.append("sort", sort);
        if (asc) {
            qc.append("asc", "true");
        }
        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            1000L,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new SearchFormatSupportCallback(context, getFields));
    }

    private final class GeoServicesOrgPhotosContext implements UniversalSearchProxyRequestContext {
        private final Map.Entry<Double, Double> coordinates;
        private final Map.Entry<Long, Long> timerange;
        private final Integer distanceThreshold;
        private final Integer dssmThreshold;
        private final Collection<String> requests;
        private final User user;
        private final AsyncClient client;
        private final ProxySession session;
        private final int length;
        private final int offset;

        public GeoServicesOrgPhotosContext(final ProxySession session) throws BadRequestException  {
            Long uid = session.params().getLong("uid");
            user = new User("disk_queue", new LongPrefix(uid));
            requests = session.params().get(
                "text",
                Collections.emptySet(),
                new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new, ','));
            dssmThreshold = session.params().getInt("dssm_threshold", 5444);
            client = proxy.searchClient().adjust(session.context());
            Double latitude = session.params().getDouble("latitude", null);
            Double longitude = session.params().getDouble("longitude", null);
            if (latitude != null && longitude != null) {
                coordinates = new AbstractMap.SimpleEntry<>(latitude, longitude);
                distanceThreshold = session.params().get("max_distance", null, PositiveIntegerValidator.INSTANCE);
            } else {
                coordinates = null;
                distanceThreshold = null;
            }

            length = session.params().get("length", 20, PositiveIntegerValidator.INSTANCE);
            offset = session.params().get("offset", 0, NonNegativeIntegerValidator.INSTANCE);

            Long timeStart = session.params().getLong("time_start", null);
            Long timeEnd = session.params().getLong("time_end", null);
            if (timeStart != null) {
                if (timeEnd == null) {
                    timeEnd = System.currentTimeMillis();
                }

                timerange = new AbstractMap.SimpleEntry<>(timeStart, timeEnd);
            } else {
                timerange = null;
            }

            if (timerange == null && coordinates == null && requests.isEmpty()) {
                throw new BadRequestException(
                    "No filters supplied, should be one or combination of timerange/coordinates/text");
            }
            this.session = session;
        }

        public Integer dssmThreshold() {
            return dssmThreshold;
        }

        public Map.Entry<Long, Long> timerange() {
            return timerange;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return Long.MAX_VALUE;
        }

        public Map.Entry<Double, Double> coordinates() {
            return coordinates;
        }

        public Integer distanceThreshold() {
            return distanceThreshold;
        }

        public Collection<String> requests() {
            return requests;
        }

        public ProxySession session() {
            return session;
        }

        public int length() {
            return length;
        }

        public int offset() {
            return offset;
        }
    }

    private static class SearchFormatSupportCallback extends AbstractProxySessionCallback<JsonObject> {
        private final Set<String> properties;

        public SearchFormatSupportCallback(
            final GeoServicesOrgPhotosContext context,
            final Set<String> properties)
        {
            super(context.session());

            this.properties = new LinkedHashSet<>(properties);
            this.properties.remove("stid");
            this.properties.remove("preview_stid");
            this.properties.remove("id");
            this.properties.remove("key");
        }

        @Override
        public void completed(final JsonObject resultObj) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                JsonMap result = resultObj.asMap();
                JsonList hits = result.getList("hitsArray");
                long hitsCount = result.getLong("hitsCount");
                writer.startObject();
//                writer.key("request");
//                writer.value("");
                writer.key("response");
                writer.startObject();
                writer.key("found");
                writer.startObject();
                writer.key("all");
                writer.value(hitsCount);
                writer.endObject();
                writer.key("results");
                writer.startArray();
                for (JsonObject hit: hits) {
                    JsonMap map = hit.asMap();
                    String id = map.getString("id");
                    writer.startObject();
                    writer.key("docs");
                    writer.value("1");
                    writer.key("attr");
                    writer.value("");
                    writer.key("groups");
                    writer.startArray();
                    writer.startObject();
                    writer.key("doccount");
                    writer.value("1");
                    writer.key("relevance");
                    writer.value(id);
                    writer.key("documents");
                    writer.startArray();
                    writer.startObject();
                    writer.key("docId");
                    writer.value(id);
                    writer.key("url");
                    writer.value("");
                    writer.key("relevance");
                    writer.value(id);
                    writer.key("properties");
                    writer.startObject();
                    writer.key("scope");
                    writer.value("file");
                    writer.key("stid");
                    writer.value(map.getString("stid"));
                    writer.key("preview_stid");
                    writer.value(map.getString("preview_stid", null));
                    writer.key("key");
                    writer.value(map.getString("key"));
                    writer.key("id");
                    writer.value(map.getString("id"));
//                    for (String property: properties) {
//                        writer.key(property);
//                        writer.value(map.getOrNull(property));
//                    }
                    writer.endObject();
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();
                }

                writer.endArray();
                writer.endObject();
                writer.endObject();
            } catch (JsonException | IOException je) {
                session.handleException(
                    new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, je));
                failed(je);
                return;
            }

            session.response(HttpStatus.SC_OK, sbw.toString());
        }
    }

//    private static class Callback extends AbstractProxySessionCallback<JsonObject> {
//        public Callback(final GeoServicesOrgPhotosContext context) {
//            super(context.session());
//        }
//
//        @Override
//        public void completed(final JsonObject resultObj) {
//            StringBuilderWriter sbw = new StringBuilderWriter();
//            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
//                JsonMap result = resultObj.asMap();
//                JsonList hits = result.getList("hitsArray");
//                writer.startObject();
//                writer.key("images");
//                writer.startArray();
//                for (JsonObject hit: hits) {
//                    JsonMap map = hit.asMap();
//                    writer.startObject();
//                    writer.key("stid");
//                    writer.value(map.getString("stid"));
//                    writer.key("preview_stid");
//                    writer.value(map.getString("preview_stid", null));
//                    writer.key("id");
//                    writer.value(map.getString("id"));
//                    writer.key("latitude");
//                    writer.value(map.getOrNull("latitude"));
//                    writer.key("longitude");
//                    writer.value(map.getOrNull("longitude"));
//                    writer.key("dssm_score");
//                    writer.value(map.getOrNull("dssmScore"));
//                    writer.key("date");
//                    writer.value(map.getOrNull("date"));
//                    writer.key("distance");
//                    writer.value(map.getOrNull("distance"));
//                    writer.key("score");
//                    writer.value(map.getOrNull("score"));
//                    writer.endObject();
//                }
//                writer.endArray();
//                writer.endObject();
//            } catch (JsonException | IOException je) {
//                failed(je);
//            }
//
//            session.response(HttpStatus.SC_OK, sbw.toString());
//        }
//    }
}
