package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

public class I2TCalcHandler implements HttpAsyncRequestHandler<JsonObject> {
    private final Proxy proxy;

    public I2TCalcHandler(final Proxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            return new JsonAsyncTypesafeDomConsumer(
                ((HttpEntityEnclosingRequest) request).getEntity(),
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            throw new BadRequestException("Payload expected");
        }
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(proxy, exchange, context);
        String response;
        try {
            response = calcI2T(payload, session);
        } catch (IOException | JsonException e) {
            throw new BadRequestException(e);
        }
        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                response,
                ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset())));
    }

    private String calcI2T(
        final JsonObject payload,
        final ProxySession session)
        throws BadRequestException, IOException, JsonException
    {
        JsonList words = payload.get("words").asList();
        JsonType jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = jsonType.create(sbw)) {
            writer.startObject();
            writer.key("results");
            writer.startArray();
            for (JsonObject word: words) {
                writer.startObject();
                String text = word.asString();
                writer.key("word");
                writer.value(text);
                try {
                    String i2t =
                        proxy.dssm().apply(text, null).process(null, 0, 0);
                    writer.key("i2t");
                    writer.value(i2t);
                } catch (JniWrapperException e) {
                    session.logger().log(
                        Level.WARNING,
                        "Failed to calc i2t for text \"" + text + '"',
                        e);
                    writer.key("error");
                    writer.value(e);
                }
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        }
        return sbw.toString();
    }
}

