package ru.yandex.search.disk.proxy;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.erratum.ErratumConfig;
import ru.yandex.erratum.ImmutableErratumConfig;
import ru.yandex.geocoder.GeocoderConfig;
import ru.yandex.geocoder.ImmutableGeocoderConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.jniwrapper.ImmutableJniWrapperConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.proxy.universal.ImmutableUniversalSearchProxyConfig;

public class ImmutableProxyConfig
    extends ImmutableUniversalSearchProxyConfig
    implements ProxyConfig
{
    private final ImmutableErratumConfig misspellConfig;
    private final ImmutableGeocoderConfig geoSearchConfig;
    private final ImmutableHttpHostConfig userSplitConfig;
    private final ImmutableHttpHostConfig faceIndexerConfig;
    private final String imagesFilter;
    private final String searchPostFilter;
    private final String diskService;
    private final String photosliceService;
    private final String ipddService;
    private final Set<String> producerServices;
    private final long fatUserDocs;
    private final ImmutableHttpTargetConfig photosliceClientConfig;
    private final ImmutableJniWrapperConfig dssmConfig;
    private final long dssmThreshold;
    private final ImmutableURIConfig djfsConfig;
    private final ImmutableTvm2ClientConfig tvm2ClientConfig;
    private final Long failoverSearchDelay;
    private final String djfsTvmClientId;
    private final String geoTvmClientId;

    public ImmutableProxyConfig(final ProxyConfig config)
        throws ConfigException
    {
        super(config);
        ErratumConfig misspellConfig = config.misspellConfig();
        if (misspellConfig == null) {
            this.misspellConfig = null;
        } else {
            this.misspellConfig = new ImmutableErratumConfig(misspellConfig);
        }
        GeocoderConfig geoSearchConfig = config.geoSearchConfig();
        if (geoSearchConfig == null) {
            this.geoSearchConfig = null;
        } else {
            this.geoSearchConfig = new ImmutableGeocoderConfig(geoSearchConfig);
        }
        imagesFilter = config.imagesFilter();
        if (imagesFilter == null || imagesFilter.isEmpty()) {
            throw new ConfigException("Images filter wasn't set");
        }

        if (config.faceIndexerConfig() != null) {
            faceIndexerConfig =
                new ImmutableHttpHostConfig(config.faceIndexerConfig());
        } else {
            faceIndexerConfig = null;
        }
        searchPostFilter = config.searchPostFilter();
        diskService = config.diskService();
        if (diskService == null || diskService.isEmpty()) {
            throw new ConfigException("Disk service name wasn't set");
        }
        photosliceService = config.photosliceService();
        if (photosliceService == null || photosliceService.isEmpty()) {
            throw new ConfigException("Photoslice service name wasn't set");
        }
        ipddService = config.ipddService();
        if (ipddService == null || ipddService.isEmpty()) {
            throw new ConfigException("IPDD service name wasn't set");
        }
        producerServices = new HashSet<>(config.producerServices());
        Set<String> unknownServices = new HashSet<>(producerServices);
        unknownServices.removeAll(
            Arrays.asList(diskService, photosliceService, ipddService));
        if (!unknownServices.isEmpty()) {
            throw new ConfigException(
                "Unknown services configured as producer services: "
                + unknownServices);
        }
        fatUserDocs = config.fatUserDocs();
        if (fatUserDocs < 0L) {
            throw new ConfigException("Fat user docs count is not set");
        }

        photosliceClientConfig =
            new ImmutableHttpTargetConfig(config.photosliceClientConfig());

        if (producerServices.isEmpty()) {
            if (producerClientConfig() != null) {
                throw new ConfigException(
                    "No producer services set, "
                    + "but producer client configured");
            }
        } else if (producerClientConfig() == null) {
            throw new ConfigException(
                "Producer services are set, "
                + "but no producer client configured");
        }

        if (config.userSplitConfig() != null) {
            userSplitConfig = new ImmutableHttpHostConfig(config.userSplitConfig());
        } else {
            userSplitConfig = null;
        }

        dssmConfig = new ImmutableJniWrapperConfig(config.dssmConfig());
        dssmThreshold = config.dssmThreshold();
        djfsConfig = new ImmutableURIConfig(config.djfsConfig());
        tvm2ClientConfig =
            new ImmutableTvm2ClientConfig(config.tvm2ClientConfig());
        failoverSearchDelay = config.failoverSearchDelay();
        djfsTvmClientId = config.djfsTvmClientId();
        geoTvmClientId = config.geoTvmClientId();
    }

    @Override
    public ImmutableErratumConfig misspellConfig() {
        return misspellConfig;
    }

    @Override
    public ImmutableGeocoderConfig geoSearchConfig() {
        return geoSearchConfig;
    }

    @Override
    public String imagesFilter() {
        return imagesFilter;
    }

    @Override
    public String searchPostFilter() {
        return searchPostFilter;
    }

    @Override
    public String diskService() {
        return diskService;
    }

    @Override
    public String photosliceService() {
        return photosliceService;
    }

    @Override
    public String ipddService() {
        return ipddService;
    }

    @Override
    public Set<String> producerServices() {
        return producerServices;
    }

    @Override
    public long fatUserDocs() {
        return fatUserDocs;
    }

    @Override
    public ImmutableHttpTargetConfig photosliceClientConfig() {
        return photosliceClientConfig;
    }

    @Override
    public ImmutableJniWrapperConfig dssmConfig() {
        return dssmConfig;
    }

    @Override
    public long dssmThreshold() {
        return dssmThreshold;
    }

    @Override
    public ImmutableURIConfig djfsConfig() {
        return djfsConfig;
    }

    @Override
    public ImmutableTvm2ClientConfig tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    @Override
    public Long failoverSearchDelay() {
        return failoverSearchDelay;
    }

    @Override
    public String djfsTvmClientId() {
        return djfsTvmClientId;
    }

    @Override
    public String geoTvmClientId() {
        return geoTvmClientId;
    }

    @Override
    public ImmutableHttpHostConfig userSplitConfig() {
        return userSplitConfig;
    }

    @Override
    public ImmutableHttpHostConfig faceIndexerConfig() {
        return faceIndexerConfig;
    }
}

