package ru.yandex.search.disk.proxy;

import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.http.proxy.HttpEntitySendingCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;

public class ListingHandler extends AbstractUidProxyHandler {
    private static final String KEY = "key";
    private static final String AND = " AND ";
    private static final String SEARCH_QUALIFIER = "search-qualifier";
    private static final int REMINDERS_IO_PRIO = 6000;

    public ListingHandler(final Proxy proxy) {
        super(proxy, proxy.diskService());
    }

    private static void and(final StringBuilder text) {
        if (text.length() != 0) {
            text.append(AND);
        }
    }

    private static void fieldFilter(
        final String name,
        final CgiParams params,
        final StringBuilder text)
        throws HttpException
    {
        List<String> values = params.getAll(
            name,
            null,
            NonEmptyValidator.INSTANCE,
            new ArrayList<String>());
        if (values != null) {
            and(text);
            text.append(name);
            text.append(':');
            text.append('(');
            for (String value: values) {
                text.append('\"');
                text.append(SearchRequestText.quoteEscape(value));
                text.append('\"');
                text.append(' ');
            }
            text.setCharAt(text.length() - 1, ')');
        }
    }

    @Override
    public void handle(
        final ProxySession session,
        final User user,
        final List<HttpHost> hosts)
        throws HttpException
    {
        CgiParams params = session.params();
        QueryConstructor query = new QueryConstructor("/search-listing?");
        query.append("prefix", user.prefix().toString());
        StringBuilder text = new StringBuilder();
        Long fromEtime =
            params.get("from-etime", null, NonNegativeLongValidator.INSTANCE);
        Long toEtime =
            params.get("to-etime", null, NonNegativeLongValidator.INSTANCE);
        if (fromEtime != null || toEtime != null) {
            if (fromEtime == null) {
                fromEtime = 0L;
            }
            if (toEtime == null) {
                toEtime = Long.MAX_VALUE;
            }
            text.append("etime:[");
            text.append(fromEtime);
            text.append(" TO ");
            text.append(toEtime);
            text.append(']');
        }
        if (params.getBoolean(Proxy.APPLY_PHOTOSLICE_FILTER, false)) {
            and(text);
            text.append('(');
            text.append(proxy.config().imagesFilter());
            text.append(')');
        }
        fieldFilter("mimetype", params, text);
        fieldFilter("mediatype", params, text);
        fieldFilter("type", params, text);
        String key;
        if (text.length() == 0) {
            key = params.get(KEY, NonEmptyValidator.INSTANCE);
        } else {
            key = params.get(KEY, null, NonEmptyValidator.INSTANCE);
        }
        if (key != null) {
            and(text);
            text.append("key:");
            text.append(
                SearchRequestText.fullEscape(key, false)
                    .replace('Ё', 'Е')
                    .replace('ё', 'е'));
            text.append('*');
        }
        query.append("text", new String(text));
        query.copyIfPresent(params, "hr");
        query.copyIfPresent(params, "json-type");
        query.copyIfPresent(params, "get");
        query.copyIfPresent(params, "skip-nulls");
        query.copyIfPresent(params, "offset");
        query.copyIfPresent(params, "length");
        query.copyIfPresent(params, "sort");
        query.copyIfPresent(params, "asc");
        query.copyIfPresent(params, "group");
        query.copyIfPresent(params, "merge_func");
        query.copyIfPresent(params, "aggregate");
        query.copyIfPresent(params, SEARCH_QUALIFIER);
        String searchQualifier = params.getString(SEARCH_QUALIFIER, null);
        if (searchQualifier != null) {
            query.append("IO_PRIO", REMINDERS_IO_PRIO);
        }
        for (String postfilter: params.getAll(Proxy.POSTFILTER)) {
            query.append(Proxy.POSTFILTER, postfilter);
        }
        AsyncClient client = proxy.searchClient().adjust(session.context());
        client.execute(
            hosts,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            NByteArrayEntityAsyncConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new HttpEntitySendingCallback(session));
    }

    @Override
    public String toString() {
        return "Files listing handler";
    }
}

