package ru.yandex.search.disk.proxy;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.parser.searchmap.User;

public class ProducerHostsResolver implements ProxyRequestHandler {
    private final ProxyHandler handler;
    private final ProducerClient producerClient;

    public ProducerHostsResolver(
        final ProxyHandler handler,
        final ProducerClient producerClient)
    {
        this.handler = handler;
        this.producerClient = producerClient;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        final User user = handler.user(session);
        producerClient
            .executeWithInfo(
                user,
                session.listener().createContextGeneratorFor(producerClient),
                new FutureCallback<List<QueueHostInfo>>() {
                    @Override
                    public void cancelled() {
                        session.logger().warning(
                            "Host resolving cancelled for user " + user
                            + '\n' + session.listener().details());
                    }

                    @Override
                    public void completed(final List<QueueHostInfo> hosts) {
                        session.logger().info(
                            "Hosts for user " + user + " are " + hosts);
                        if (hosts.isEmpty()) {
                            session.handleException(
                                new NotFoundException(
                                    "Don't know where the user " + user));
                        } else {
                            List<HttpHost> httpHosts = new ArrayList<>(hosts.size());
                            for (QueueHostInfo info: hosts) {
                                httpHosts.add(info.host());
                            }
                            try {
                                handler.handle(
                                    session,
                                    user,
                                    Proxy.shuffle(httpHosts, user));
                            } catch (HttpException e) {
                                session.handleException(e);
                            }
                        }
                    }

                    @Override
                    public void failed(final Exception e) {
                        session.logger().log(
                            Level.WARNING,
                            "Host resolving failed for user " + user
                            + '\n' + session.listener().details(),
                            e);
                        session.handleException(
                            HttpExceptionConverter.toHttpException(e));
                    }
                });
    }

    @Override
    public String toString() {
        return handler.toString();
    }
}

