package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ImmutableErratumConfig;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.geocoder.GeocoderClient;
import ru.yandex.geocoder.ImmutableGeocoderConfig;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.face.FaceClusterResourcesHandler;
import ru.yandex.search.disk.proxy.face.FaceClustersHandler;
import ru.yandex.search.disk.proxy.face.FaceDeltasHandler;
import ru.yandex.search.disk.proxy.face.FaceReindexHandler;
import ru.yandex.search.disk.proxy.ipdd.IpddClusterizeHandler;
import ru.yandex.search.disk.proxy.ipdd.IpddCountersHandler;
import ru.yandex.search.disk.proxy.ipdd.IpddGroupHandler;
import ru.yandex.search.disk.proxy.ipdd.IpddSearchHandler;
import ru.yandex.search.disk.proxy.suggest.PromoSuggestHandler;
import ru.yandex.search.disk.proxy.suggest.SuggestHandler;
import ru.yandex.search.disk.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class Proxy extends UniversalSearchProxy<ImmutableProxyConfig> {
  public static final String POSTFILTER = "postfilter";
  public static final String APPLY_PHOTOSLICE_FILTER =
      "apply-photoslice-filter";
  public static final String DISK_ALLOWED_ROOTS =
      "attach,disk,fotki,hidden,lnarod,mail,mulca,narod,photounlim,client,"
          + "photostream,settings,share,trash,yareader,yavideo,notes";

  private static final String RESOURCE_ID = "resource_id";
  private static final Long ZERO = 0L;
  private static final Long ONE = 1L;

  private final TimeFrameQueue<Map<String, long[]>> scopes;
  private final TimeFrameQueue<Long> emptySearches;
  private final TimeFrameQueue<Long> historySuggestIndexations;
  private final TimeFrameQueue<Long> hnswCalls;
  private final String searchPostFilter;
  private final ErratumClient misspellClient;
  private final GeocoderClient geoSearchClient;
  private final String clusterizeQuery;
  private final String diskService;
  private final String photosliceService;
  private final String ipddService;
  private final Set<String> producerServices;
  private final AsyncClient photosliceClient;
  private final JniWrapper dssm;
  private final long dssmThreshold;
  private final AsyncClient djfsClient;
  private final AsyncClient uaasClient;
  private final Tvm2TicketRenewalTask tvm2RenewalTask;
  private final UpstreamStater historySuggestIndexationStater;
  private final List<Warmable> warmables;

  // CSOFF: MethodLength
  public Proxy(final ImmutableProxyConfig config)
      throws GeneralSecurityException,
      HttpException,
      IOException,
      JniWrapperException,
      JsonException,
      URISyntaxException {
    super(config);
    this.warmables = new ArrayList<>();

    scopes = new TimeFrameQueue<>(config.metricsTimeFrame());
    emptySearches = new TimeFrameQueue<>(config.metricsTimeFrame());
    historySuggestIndexations =
        new TimeFrameQueue<>(config.metricsTimeFrame());

    hnswCalls = new TimeFrameQueue<>(config.metricsTimeFrame());
    registerStater(
            new PassiveStaterAdapter<>(
                    hnswCalls,
                    new NamedStatsAggregatorFactory<>(
                            "hnsw-search-requests_ammm",
                            IntegralSumAggregatorFactory.INSTANCE)));

    MultiSearchStater multiSearchStater =
        new MultiSearchStater(config.metricsTimeFrame());
    searchPostFilter = config.searchPostFilter();
    ImmutableErratumConfig misspellConfig = config.misspellConfig();
    if (misspellConfig == null) {
      misspellClient = null;
    } else {
      misspellClient =
          registerClient(
              "Erratum",
              new ErratumClient(reactor, misspellConfig),
              misspellConfig);
    }

    if (config.userSplitConfig() != null) {
      uaasClient = client("UaasClient", config.userSplitConfig());
    } else {
      uaasClient = null;
    }
    ImmutableGeocoderConfig geoSearchConfig =
        config.geoSearchConfig();
    if (geoSearchConfig == null) {
      geoSearchClient = null;
    } else {
      geoSearchClient =
          registerClient(
              "Geo",
              new GeocoderClient(reactor, geoSearchConfig),
              geoSearchConfig);
    }
    PctEncoder encoder = new PctEncoder(PctEncodingRule.QUERY);
    encoder.process(config.imagesFilter().toCharArray());
    clusterizeQuery =
        "/clusterize-photoslice?merged-length=40000&text=" + encoder;
    diskService = config.diskService();
    photosliceService = config.photosliceService();
    ipddService = config.ipddService();
    producerServices = config.producerServices();
    photosliceClient =
        client("Photoslice", config.photosliceClientConfig());
    long start = System.currentTimeMillis();
    logger().info("Loading DSSM");
    dssm =
        JniWrapper.create(
            config.dssmConfig(),
            new ThreadFactoryConfig(config.name() + "-Jni-")
                .group(getThreadGroup())
                .daemon(true));
    closeChain.add(new GenericNonThrowingCloseableAdapter<>(dssm));
    logger().info(
        "DSSM loading completed in " + (System.currentTimeMillis() - start)
            + " ms");
    dssmThreshold = config.dssmThreshold();
    djfsClient = client("DJFS", config.djfsConfig());
    tvm2RenewalTask = new Tvm2TicketRenewalTask(
        logger().addPrefix("tvm2"),
        serviceContextRenewalTask,
        config.tvm2ClientConfig());

    historySuggestIndexationStater = new UpstreamStater(
        config.metricsTimeFrame(),
        "history-suggest-indexation");
    registerStater(historySuggestIndexationStater);

    register(
        new Pattern<>("/hnsw-search", true),
        wrapWithHostsResolver(
            new HnswSearchHandler(this, multiSearchStater)),
        RequestHandlerMapper.GET);

    register(
        new Pattern<>("/clusterize", false),
        wrapWithHostsResolver(new ClusterizeHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/folder-size", false),
        wrapWithHostsResolver(new FolderSizeHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/listing", false),
        wrapWithHostsResolver(new ListingHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/get", false),
        wrapWithHostsResolver(new GetHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/ipdd/clusterize", false),
        wrapWithHostsResolver(new IpddClusterizeHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/ipdd/counters", false),
        wrapWithHostsResolver(new IpddCountersHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/ipdd/group", false),
        wrapWithHostsResolver(new IpddGroupHandler(this)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/ipdd/search", false),
        wrapWithHostsResolver(new IpddSearchHandler(this, false)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/ipdd/search/raw", false),
        wrapWithHostsResolver(new IpddSearchHandler(this, true)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/drop-index", false),
        new DropIndexHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("", true),
        wrapWithHostsResolver(
            new SearchHandler(this, multiSearchStater)),
        RequestHandlerMapper.GET);

    // Suggest handlers
    register(
        new Pattern<>("/suggest/promo", false),
        new PromoSuggestHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/suggest", false),
        new SuggestHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/folder-resources", false),
        new FolderResourcesHandler(this),
        RequestHandlerMapper.GET);
    // faces
    register(
        new Pattern<>("/api/faces/clusters", false),
        new FaceClustersHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/api/faces/cluster-photos", false),
        new FaceClusterResourcesHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/api/faces/deltas", false),
        new FaceDeltasHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/api/geo/org_photos", false),
        new GeoServicesOrgPhotosHandler(this, dssm),
        RequestHandlerMapper.GET);

    // Hackaton handlers
    register(
        new Pattern<>("/random-photo", false),
        new RandomPhotoHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/all-geos", false),
        new AllGeosHandler(this),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/photos-date-range", false),
        new PhotosDateRangeHandler(this),
        RequestHandlerMapper.GET);

    // PS-3122
    register(
        new Pattern<>("/i2t", false),
        new I2TCalcHandler(this),
        RequestHandlerMapper.POST);

    register(
        new Pattern<>("/warmup", false),
        wrapWithHostsResolver(
            new WarmupHandler(this, warmables)),
        RequestHandlerMapper.GET);
    register(
        new Pattern<>("/face/reindex", false),
        new FaceReindexHandler(this),
        RequestHandlerMapper.GET);

    registerStater(new ScopesStater(scopes));
    registerStater(
        new PassiveStaterAdapter<>(
            emptySearches,
            new DuplexStaterFactory<>(
                new NamedStatsAggregatorFactory<>(
                    "empty-searches_ammm",
                    IntegralSumAggregatorFactory.INSTANCE),
                new NamedStatsAggregatorFactory<>(
                    "empty-searches-count_ammm",
                    CountAggregatorFactory.INSTANCE))));
    registerStater(
        new PassiveStaterAdapter<>(
            historySuggestIndexations,
            new DuplexStaterFactory<>(
                new NamedStatsAggregatorFactory<>(
                    "history-suggest-indexed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE),
                new NamedStatsAggregatorFactory<>(
                    "history-suggest-requests_ammm",
                    CountAggregatorFactory.INSTANCE))));
    registerStater(multiSearchStater);
  }

  // CSON: MethodLength

  @Override
  public void start() throws IOException {
    tvm2RenewalTask.start();
    super.start();
  }

  @Override
  public void close() throws IOException {
    tvm2RenewalTask.cancel();
    super.close();
  }

  @Override
  public HttpAsyncRequestHandler<?> register(
      final Pattern<RequestInfo> pattern,
      final ProxyRequestHandler handler,
      final String method) {
    if (handler instanceof Warmable) {
      this.warmables.add((Warmable) handler);
    }
    return super.register(pattern, handler, method);
  }

  private ProxyRequestHandler wrapWithHostsResolver(
      final ProxyHandler handler) {
    if (handler instanceof Warmable) {
      this.warmables.add((Warmable) handler);
    }

    if (producerServices.contains(handler.serviceName())) {
      return new ProducerHostsResolver(handler, producerClient());
    } else {
      return new SearchMapHostsResolver(handler, searchMap());
    }
  }

  public String searchPostFilter() {
    return searchPostFilter;
  }

  public ErratumClient misspellClient() {
    return misspellClient;
  }

  public GeocoderClient geoSearchClient() {
    return geoSearchClient;
  }

  public String clusterizeQuery() {
    return clusterizeQuery;
  }

  public String diskService() {
    return diskService;
  }

  public String photosliceService() {
    return photosliceService;
  }

  public String ipddService() {
    return ipddService;
  }

  public AsyncClient photosliceClient() {
    return photosliceClient;
  }

  public JniWrapper dssm() {
    return dssm;
  }

  public long dssmThreshold() {
    return dssmThreshold;
  }

  public UpstreamStater historySuggestIndexationStater() {
    return historySuggestIndexationStater;
  }

  public String djfsTvm2Ticket() {
    return tvm2RenewalTask.ticket(config.djfsTvmClientId());
  }

  public String geoTvm2Ticket() {
    return tvm2RenewalTask.ticket(config.geoTvmClientId());
  }

  public void searchCompleted(
      final Map<String, long[]> scopes,
      final boolean empty) {
    this.scopes.accept(scopes);
    if (empty) {
      emptySearches.accept(ONE);
    } else {
      emptySearches.accept(ZERO);
    }
  }

  public void historySuggestIndexed() {
    historySuggestIndexations.accept(ONE);
  }

  public void historySuggestRemoved() {
    historySuggestIndexations.accept(ZERO);
  }

  public void filterResources(
      final SuggestRequestContext context,
      final JsonList hits,
      final FutureCallback<? super Map<String, JsonMap>> callback)
      throws BadRequestException, JsonException {
    StringBuilder sb =
        new StringBuilder(config.djfsConfig().uri().toASCIIString());
    sb.append(config.djfsConfig().firstCgiSeparator());
    sb.append("uid=");
    sb.append(context.user().prefix());
    boolean empty = true;
    QueryConstructor query = new QueryConstructor(sb);
    for (JsonObject hit : hits) {
      String resourceId = hit.get(RESOURCE_ID).asStringOrNull();
      if (resourceId != null) {
        empty = false;
        query.append(RESOURCE_ID, resourceId);
      }
    }
    if (empty) {
      callback.completed(Collections.emptyMap());
    } else {
      HeaderAsyncRequestProducerSupplier request;
      try {
        request =
            new HeaderAsyncRequestProducerSupplier(
                new AsyncGetURIRequestProducerSupplier(
                    query.toString()),
                new BasicHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    djfsTvm2Ticket()));
      } catch (URISyntaxException e) {
        throw new BadRequestException(e);
      }

      AsyncClient client =
          djfsClient.adjust(context.session().context());
      client.execute(
          request,
          JsonAsyncTypesafeDomConsumerFactory.OK,
          context.session().listener().createContextGeneratorFor(client),
          new CollectResourcesCallback(callback));
    }
  }

  private static class CollectResourcesCallback
      extends AbstractFilterFutureCallback<JsonObject, Map<String, JsonMap>> {
    CollectResourcesCallback(
        final FutureCallback<? super Map<String, JsonMap>> callback) {
      super(callback);
    }

    @Override
    public void completed(final JsonObject response) {
      try {
        JsonList items = response.get("items").asList();
        Map<String, JsonMap> resources =
            new HashMap<>(items.size() << 1);
        for (JsonObject item : items) {
          JsonMap resource = item.asMap();
          resources.put(
              resource.get(RESOURCE_ID).asString(),
              resource);
        }
        callback.completed(resources);
      } catch (JsonException e) {
        failed(new ServiceUnavailableException(e));
      }
    }
  }

  private static class ScopesStater implements Stater {
    private final TimeFrameQueue<Map<String, long[]>> scopes;

    ScopesStater(final TimeFrameQueue<Map<String, long[]>> scopes) {
      this.scopes = scopes;
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E {
      IdentityHashMap<String, long[]> scopes = new IdentityHashMap<>();
      for (String scope : SearchHandler.SCOPES) {
        scopes.put(scope, new long[1]);
      }
      long searches = 0L;
      for (Map<String, long[]> stat : this.scopes) {
        ++searches;
        for (Map.Entry<String, long[]> entry : stat.entrySet()) {
          scopes.get(entry.getKey())[0] += entry.getValue()[0];
        }
      }
      for (Map.Entry<String, long[]> entry : scopes.entrySet()) {
        statsConsumer.stat(
            StringUtils.concat(
                "scope-results-",
                entry.getKey(),
                "_ammm"),
            entry.getValue()[0]);
      }
      statsConsumer.stat("searches-count_ammm", searches);
    }
  }

  public static void addFastMovedDp(
      final QueryConstructor query)
      throws BadRequestException {
    query.append(
        "dp",
        "tree_root(parent_fid,fid,type:dir,disk_concat(/),name key)");
    query.append(
        "disk-allowed-roots",
        DISK_ALLOWED_ROOTS);
  }

  public AsyncClient uaasClient() {
    return uaasClient;
  }

  public TimeFrameQueue<Long> hnswCalls() {
    return hnswCalls;
  }
}

