package ru.yandex.search.disk.proxy;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.ConcatSupplierFactory;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.disk.proxy.rules.DocumentsSearchRule;
import ru.yandex.search.disk.proxy.rules.DssmSearchRule;
import ru.yandex.search.disk.proxy.rules.FileSearchRule;
import ru.yandex.search.disk.proxy.rules.FoldersSearchRule;
import ru.yandex.search.disk.proxy.rules.GeoSearchRule;
import ru.yandex.search.disk.proxy.rules.MultiSearchRule;
import ru.yandex.search.disk.proxy.rules.NameKeywordSearchRule;
import ru.yandex.search.disk.proxy.rules.NamedRule;
import ru.yandex.search.disk.proxy.rules.OptionalHnswSearchRule;
import ru.yandex.search.disk.proxy.rules.PlainSearchRule;
import ru.yandex.search.disk.proxy.rules.SetScopeRule;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.BasicSearchRequest;
import ru.yandex.search.rules.MisspellRule;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRule;

public class SearchHandler implements ProxyHandler, PrintkeysWarmable {
  private static final String HOW = "how";
  private static final String FOLDER = "folder";
  private static final String NAME = "name";
  private static final String TAGS = "tags";
  private static final String META = "meta";
  private static final String CV = "cv";
  private static final String OCR = "ocr";
  private static final String BODY = "body";
  private static final String EXTERNAL_HOST = "external_host";
  private static final String EXTERNAL_URL = "external_url";
  private static final String GEO = "geo";
  private static final String TRASH = "trash";

  public static final List<String> SCOPES =
      Collections.unmodifiableList(
          Arrays.asList(
              FOLDER,
              NAME,
              TAGS,
              META,
              CV,
              OCR,
              BODY,
              EXTERNAL_HOST,
              EXTERNAL_URL,
              GEO,
              TRASH
          ));

  private final Proxy proxy;
  private final String serviceName;
  private final PrefixType prefixType;
  private final SearchRule<SearchResult, DiskRequestParams, SearchInfo> rule;
  private final Set<String> warmupKeyFields;
  private final NamedRule i2tRule;

  // CSOFF: MethodLength
  protected SearchHandler(
      final Proxy proxy,
      final TimeFrameQueue<MultiSearchStat> rulesStats
  ) {
    this.proxy = proxy;
    serviceName = proxy.diskService();
    prefixType = proxy.searchMap().prefixType(serviceName);
    int seed = 0;
    List<NamedRule> rules = new ArrayList<>();
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FoldersSearchRule(new PlainSearchRule(proxy, seed++)),
                FOLDER),
            FOLDER));

    List<String> fields = new ArrayList<>();
    fields.add("name_tokenized");

    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    new FieldsTermsSupplierFactory(
                        new ArrayList<>(fields))),
                NAME),
            "name-tokenized"));

    rules.add(
        new NamedRule(
            new SetScopeRule(
                new NameKeywordSearchRule(new PlainSearchRule(proxy, seed++)),
                NAME),
            NAME));

    fields.add("fotki_tags");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    new FieldsTermsSupplierFactory(
                        new ArrayList<>(fields))),
                TAGS),
            TAGS));

    fields.add("album");
    fields.add("artist");
    fields.add("author");
    fields.add("comment");
    fields.add("composer");
    fields.add("description");
    fields.add("genre");
    fields.add("keywords");
    fields.add("manufacturer");
    fields.add("model");
    fields.add("subject");
    fields.add("title");
    FieldsTermsSupplierFactory metaFieldsTermsSupplier =
        new FieldsTermsSupplierFactory(new ArrayList<>(fields));
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    metaFieldsTermsSupplier),
                META),
            META));

    i2tRule = new NamedRule(
        new SetScopeRule(
            new DssmSearchRule(
                getInnerDssmSearchRule(proxy, seed++),
                proxy.dssm(),
                proxy.dssmThreshold()),
            CV),
        "i2t");
    rules.add(i2tRule);

    fields.add("ocr_text");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                OCR),
            OCR));

    fields.add("body_text");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                BODY),
            BODY));

    fields.add(EXTERNAL_HOST);
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                EXTERNAL_HOST),
            "external-host"));

    fields.add(EXTERNAL_URL);
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                EXTERNAL_URL),
            "external-url"));

    if (proxy.geoSearchClient() != null) {
      rules.add(
          new NamedRule(
              new SetScopeRule(
                  new GeoSearchRule(
                      proxy,
                      new PlainSearchRule(proxy, seed++)),
                  GEO),
              GEO));
    }

    SearchRule<SearchResult, DiskRequestParams, SearchInfo> rule = new DocumentsSearchRule(
        new MisspellRule<>(
            new MultiSearchRule(rules, rulesStats),
            proxy.misspellClient(),
            "text",
            x -> x.hitsCount() == 0L),
        i2tRule,
        new SetScopeRule(new PlainSearchRule(proxy, seed++), NAME),
        proxy);

    this.rule = rule;

    warmupKeyFields = new LinkedHashSet<>(fields);
  }

  public SearchRule<SearchResult, DiskRequestParams, SearchInfo> createOrderExpRule(
      final TimeFrameQueue<MultiSearchStat> rulesStats) {
    int seed = 0;
    List<NamedRule> rules = new ArrayList<>();
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FoldersSearchRule(new PlainSearchRule(proxy, seed++)),
                FOLDER),
            FOLDER));

    List<String> fields = new ArrayList<>();
    fields.add("name_tokenized");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    new FieldsTermsSupplierFactory(
                        new ArrayList<>(fields))),
                NAME),
            "name-tokenized"));

    rules.add(
        new NamedRule(
            new SetScopeRule(
                new NameKeywordSearchRule(
                    new PlainSearchRule(proxy, seed++)),
                NAME),
            NAME));

    rules.add(
        new NamedRule(
            new SetScopeRule(
                new DssmSearchRule(
                    getInnerDssmSearchRule(proxy, seed++),
                    proxy.dssm(),
                    proxy.dssmThreshold()),
                CV),
            "i2t"));

    fields.add("fotki_tags");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    new FieldsTermsSupplierFactory(
                        new ArrayList<>(fields))),
                TAGS),
            TAGS));

    fields.add("album");
    fields.add("artist");
    fields.add("author");
    fields.add("comment");
    fields.add("composer");
    fields.add("description");
    fields.add("genre");
    fields.add("keywords");
    fields.add("manufacturer");
    fields.add("model");
    fields.add("subject");
    fields.add("title");
    FieldsTermsSupplierFactory metaFieldsTermsSupplier =
        new FieldsTermsSupplierFactory(new ArrayList<>(fields));
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    metaFieldsTermsSupplier),
                META),
            META));

    fields.add("ocr_text");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                OCR),
            OCR));

    fields.add("body_text");
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                BODY),
            BODY));

    fields.add(EXTERNAL_HOST);
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                EXTERNAL_HOST),
            "external-host"));

    fields.add(EXTERNAL_URL);
    rules.add(
        new NamedRule(
            new SetScopeRule(
                new FileSearchRule(
                    new PlainSearchRule(proxy, seed++),
                    new ConcatSupplierFactory<>(
                        Arrays.asList(
                            metaFieldsTermsSupplier,
                            new FieldsTermsSupplierFactory(
                                new ArrayList<>(fields))))),
                EXTERNAL_URL),
            "external-url"));

    if (proxy.geoSearchClient() != null) {
      rules.add(
          new NamedRule(
              new SetScopeRule(
                  new GeoSearchRule(
                      proxy,
                      new PlainSearchRule(proxy, seed++)),
                  GEO),
              GEO));
    }

    SearchRule<SearchResult, DiskRequestParams, SearchInfo> rule = new MisspellRule<>(
        new MultiSearchRule(rules, rulesStats),
        proxy.misspellClient(),
        "text",
        x -> x.hitsCount() == 0L);

    return rule;
  }
  // CSON: MethodLength

  protected SearchRule<SearchResult, DiskRequestParams, SearchInfo> getInnerDssmSearchRule(Proxy proxy, int seed) {
    return new OptionalHnswSearchRule(proxy, seed);
  }

  @Override
  public Set<String> warmUpKeyFields() {
    return warmupKeyFields;
  }

  @Override
  public Set<String> warmUpValueFields() {
    return new LinkedHashSet<>(Arrays.asList("id", "key"));
  }

  @Override
  public String warmupPostfix() {
    return "-search";
  }

  @Override
  public String serviceName() {
    return serviceName;
  }

  @Override
  public User user(final ProxySession session) throws HttpException {
    return new User(serviceName, session.params().get("kps", prefixType));
  }

  @Override
  public void handle(
      final ProxySession session,
      final User user,
      final List<HttpHost> hosts)
      throws HttpException {
    if (session.params().getString(HOW, null) == null) {
      session.params().replace(HOW, "mtime");
    }
    rule.execute(
        new BasicSearchRequest<>(
            session,
            new DiskSearchService(proxy, user, hosts)));
  }

  @Override
  public String toString() {
    return "Search for user files on Yandex.Disk";
  }
}

