package ru.yandex.search.disk.proxy;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.base64.Base64;
import ru.yandex.base64.Base64Decoder;
import ru.yandex.charset.Decoder;
import ru.yandex.function.StringBuilderable;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;

public class UserSplit implements StringBuilderable {
    private final String configVersion;
    private final Set<String> tests;
    private final List<JsonObject> flags;

    public UserSplit(
        final String config,
        final String tests,
        final String flags)
        throws IOException
    {
        this.tests = new LinkedHashSet<>();
        this.configVersion = config;

        for (String test: tests.split(";")) {
            String text = test.trim();

            if (!text.isEmpty()) {
                String pTest = parseTest(text);
                if (pTest != null) {
                    this.tests.add(parseTest(text));
                }
            }
        }

        String[] flagsSplit = flags.split(",");

        List<JsonObject> flagsList = new ArrayList<>(flagsSplit.length);

        for (int i = 0; i < flagsSplit.length; i++) {
            Base64Decoder base64Decoder = new Base64Decoder(Base64.URL);
            Decoder decoder = new Decoder(StandardCharsets.UTF_8);
            base64Decoder.process(flagsSplit[i].toCharArray());
            base64Decoder.processWith(decoder);

            try {
                flagsList.add(TypesafeValueContentHandler.parse(decoder.toString()));
            } catch (JsonException je) {
                throw new IOException("Failed to parse flags into json " + flagsSplit[i], je);
            }
        }

        this.flags = flagsList;
    }

    public Set<String> tests() {
        return tests;
    }

    protected String parseTest(final String test)
        throws IOException
    {
        String[] split = test.split(",");
        if (split.length == 0) {
            return null;
        }

        if (split.length != 3) {
            throw new IOException(
                "Malformed test triple " + test);
        }

        return split[0];
    }

    @Override
    public void toStringBuilder(final StringBuilder builder) {
        int i = 0;
        for (String test : tests) {
            if (i != 0) {
                builder.append(',');
            }

            builder.append(test);
            i++;
        }
    }

    public String testsToString() {
        StringBuilder builder = new StringBuilder();
        toStringBuilder(builder);
        return builder.toString();
    }

    public List<JsonObject> flags() {
        return flags;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("ConfigVersion:");
        sb.append(configVersion);
        sb.append("Experiments:");
        sb.append(testsToString());
        sb.append("Flags:");
        sb.append(JsonType.NORMAL.toString(flags));
        return sb.toString();
    }
}
