package ru.yandex.search.disk.proxy.face;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.DiskBackendFields;
import ru.yandex.ps.disk.search.FaceBackendFields;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class FaceClusterResourcesHandler implements ProxyRequestHandler {
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private static final Set<String> DEFAULT_GET_FIELDS =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(
            Arrays.asList(
                FaceBackendFields.FACE_ID.stored(),
                FaceBackendFields.FACE_CLUSTER_ID.stored(),
                FaceBackendFields.FACE_COORD_X.stored(),
                FaceBackendFields.FACE_COORD_Y.stored(),
                FaceBackendFields.FACE_WIDTH.stored(),
                FaceBackendFields.FACE_HEIGHT.stored(),
                DiskBackendFields.WIDTH.stored(),
                DiskBackendFields.HEIGHT.stored())));

    private static final Set<String> RESOURCE_GET_FIELDS = Collections.emptySet();

//        StringUtils.join(
//            Arrays.asList(
//                FaceBackendFields.FACE_CLUSTER_ID.stored(),
//                FaceBackendFields.FACE_CLUSTER_VERSION.stored()),
//            ',');

    private static final long FAILOVER_DELAY = 300L;

    private final String faceQueue;
    private final Proxy proxy;

    public FaceClusterResourcesHandler(final Proxy proxy) {
        this.proxy = proxy;
        this.faceQueue = System.getProperty("FACE_QUEUE", "disk_queue");
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Context context = new Context(session);

        Set<String> getFields = new LinkedHashSet<>(context.getFields());

        StringBuilder sb = new StringBuilder();
        sb.append(DiskBackendFields.TYPE.prefixed());
        sb.append(":face");
        sb.append(" AND ");
        sb.append(FaceBackendFields.FACE_CLUSTER_ID.prefixed());
        sb.append(":");
        sb.append(context.clusterId());
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());
        qc.append("service", context.user().service());
        qc.append("text", sb.toString());

        if (!context.resGetFields().isEmpty()) {
            if (!getFields.contains("*")) {
                getFields.add(FaceBackendFields.FACE_RESOURCE_ID.stored());
                getFields.addAll(context.resGetFields());
            }

            StringBuilder joinSb = new StringBuilder();
            joinSb.append("left_join(");
            joinSb.append(FaceBackendFields.FACE_RESOURCE_ID.stored());
            joinSb.append(',');
            joinSb.append("resource_id");
            joinSb.append(",,");
            for (String field: context.resGetFields()) {
                joinSb.append(field);
                joinSb.append(' ');
                joinSb.append(field);
                joinSb.append(',');
            }
            joinSb.setLength(joinSb.length() - 1);
            joinSb.append(')');
            qc.append("dp", joinSb.toString());
            // drop all record if right part is null
            qc.append("keep-right-null", "false");
        }

        qc.append("get", StringUtils.join(getFields, ','));

//        context.client().execute(
//            proxy.searchMap().searchHosts(context.user),
//            new BasicAsyncRequestProducerGenerator(qc.toString()),
//            System.currentTimeMillis() + 1000,
//            FAILOVER_DELAY,
//            JsonAsyncTypesafeDomConsumerFactory.OK,
//            context.session().listener().adjustContextGenerator(
//                context.client().httpClientContextGenerator()),
//            new Callback(session, context));

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new Callback(session, context));
    }

    private static final class Callback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Context context;

        public Callback(
            final ProxySession session,
            final Context context)
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            StringBuilderWriter sbw = new StringBuilderWriter();

            try (JsonWriter writer = context.jsonType().create(sbw)) {
                JsonMap result = resultObj.asMap();
                JsonList results = result.getList("hitsArray");

                writer.startObject();
                writer.key("cluster_id");
                writer.value(context.clusterId());
                writer.key("items");
                writer.startArray();
                int skippedHw = 0;
                for (JsonObject itemObj: results) {
                    JsonMap map = itemObj.asMap();

                    JsonObject width = map.get(FaceBackendFields.FACE_RESOURCE_WIDTH.stored());
                    if (width == JsonNull.INSTANCE) {
                        width = map.get(DiskBackendFields.WIDTH.stored());
                    }
                    JsonObject height = map.get(FaceBackendFields.FACE_RESOURCE_HEIGHT.stored());
                    if (height == JsonNull.INSTANCE) {
                        height = map.get(DiskBackendFields.HEIGHT.stored());
                    }

                    if (context.filterOnEmptyWidthHeight()
                        && (width == JsonNull.INSTANCE || height == JsonNull.INSTANCE))
                    {
                        skippedHw += 1;
                        if (skippedHw == 1) {
                            session.logger().info("Skipping " + JsonType.NORMAL.toString(map));
                        }
                    }

                    map.put(DiskBackendFields.WIDTH.stored(), width);
                    map.put(DiskBackendFields.HEIGHT.stored(), height);

                    writer.startObject();
                    for (String field: context.printFields()) {
                        JsonObject value = map.get(field);
                        if (value != JsonNull.INSTANCE) {
                            writer.key(field);
                            writer.value(value);
                        }
                    }

                    writer.endObject();
                }

                session.logger().warning("Skipped missing height or width " + skippedHw);
                writer.endArray();
                writer.endObject();
            } catch (IOException | JsonException e) {
                failed(e);
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }

    private final class Context implements UniversalSearchProxyRequestContext {
        private final Long uid;
        private final String clusterId;
        private final User user;
        private final boolean allowLaggingHosts;

        private final ProxySession session;
        private final AsyncClient client;
        private final JsonType jsonType;
        private final Collection<String> getFields;
        private final Collection<String> printFields;
        private final Collection<String> resGetFields;
        private final boolean filterOnEmptyWidthHeight;

        public Context(final ProxySession session) throws BadRequestException {
            this.session = session;
            this.uid = session.params().getLong("uid");
            this.clusterId = session.params().getString("cluster_id");
            this.user = new User(faceQueue, new LongPrefix(uid));
            allowLaggingHosts =
                session.params().getBoolean("allow-lagging-hosts", false);
            client = proxy.searchClient().adjust(session.context());
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            Set<String> resGetFields = session.params().get("resource_get", RESOURCE_GET_FIELDS, SET_PARSER);

            Set<String> getFields = session.params().get("get", DEFAULT_GET_FIELDS, SET_PARSER);
            printFields = new LinkedHashSet<>(getFields);
            printFields.addAll(resGetFields);

            if (!resGetFields.isEmpty()) {
                getFields = new LinkedHashSet<>(getFields);
                getFields.addAll(resGetFields);

                if (resGetFields.contains(DiskBackendFields.WIDTH.stored())) {
                    getFields.add(FaceBackendFields.FACE_RESOURCE_WIDTH.stored());
                }

                if (resGetFields.contains(DiskBackendFields.HEIGHT.stored())) {
                    getFields.add(FaceBackendFields.FACE_RESOURCE_HEIGHT.stored());
                }
            }

            this.resGetFields = resGetFields;
            this.getFields = getFields;

            filterOnEmptyWidthHeight = session.params().getBoolean("filter_empty_width_height", true);
        }

        public Collection<String> printFields() {
            return printFields;
        }

        public boolean filterOnEmptyWidthHeight() {
            return filterOnEmptyWidthHeight;
        }

        public Collection<String> getFields() {
            return getFields;
        }

        public Collection<String> resGetFields() {
            return resGetFields;
        }

        public String clusterId() {
            return clusterId;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public Long uid() {
            return uid;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE : 0L;
        }
    }
}
