package ru.yandex.search.disk.proxy.face;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.DiskBackendFields;
import ru.yandex.ps.disk.search.FaceBackendFields;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class FaceClustersHandler implements ProxyRequestHandler {
//    private static final Set<String> DEFAULT_GET = Collections.singleton("resource_id");
//    private static final CollectionParser<String, Set<String>, Exception>
//        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private static final String GET_FIELDS =
        StringUtils.join(
            Arrays.asList(
                FaceBackendFields.FACECLUSTER_ID.stored(),
                FaceBackendFields.FACECLUSTER_VERSION.stored()),
            ',');

    private static final long FAILOVER_DELAY = 300L;

    private final String faceQueue;
    private final Proxy proxy;

    public FaceClustersHandler(final Proxy proxy) {
        this.proxy = proxy;

        this.faceQueue = System.getProperty("FACE_QUEUE", "disk_queue");
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Context context = new Context(session);

        StringBuilder sb = new StringBuilder();
        sb.append(DiskBackendFields.TYPE.prefixed());
        sb.append(':');
        sb.append("face_cluster");
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());
        qc.append("service", context.user().service());
        qc.append("text", sb.toString());
        qc.append("get", GET_FIELDS);

        session.logger().info("Hosts: " + proxy.searchMap().searchHosts(context.user));

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new Callback(session, context));
    }

    private static final class Callback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Context context;

        public Callback(
            final ProxySession session,
            final Context context)
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            context.logger().info("Backend responded " + JsonType.NORMAL.toString(resultObj));
            List<FaceCluster> clusters;
            long maxVersion = 0;
            try {
                JsonMap result = resultObj.asMap();
                JsonList results = result.getList("hitsArray");
                clusters = new ArrayList<>(results.size());
                for (JsonObject itemObj: results) {
                    JsonMap itemMap = itemObj.asMap();
                    String id = itemMap.getString(FaceBackendFields.FACECLUSTER_ID.stored());
                    long version = itemMap.getLong(FaceBackendFields.FACECLUSTER_VERSION.stored());
                    clusters.add(new FaceCluster(id, version));
                    if (version > maxVersion) {
                        maxVersion = version;
                    }
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType().create(sbw)) {
                writer.startObject();
                writer.key("version");
                writer.value(maxVersion);
                writer.key("items");
                writer.value(clusters);
                writer.endObject();
            } catch (IOException e) {
                failed(e);
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }

    private final class Context implements UniversalSearchProxyRequestContext {
        private final Long uid;
        private final User user;
        private final boolean allowLaggingHosts;

        private final ProxySession session;
        private final AsyncClient client;
        private final JsonType jsonType;

        public Context(final ProxySession session) throws BadRequestException {
            this.session = session;
            this.uid = session.params().getLong("uid");
            this.user = new User(faceQueue, new LongPrefix(uid));
            allowLaggingHosts =
                session.params().getBoolean("allow-lagging-hosts", false);
            client = proxy.searchClient().adjust(session.context());
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public Long uid() {
            return uid;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE : 0L;
        }
    }

    private static final class FaceCluster implements JsonValue {
        private final String id;
        private final long version;

        public FaceCluster(final String id, final long version) {
            this.id = id;
            this.version = version;
        }

        public String id() {
            return id;
        }

        public long version() {
            return version;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("id");
            writer.value(id);
            writer.endObject();
        }
    }
}
