package ru.yandex.search.disk.proxy.face;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.disk.search.face.ClusterModification;
import ru.yandex.disk.search.face.DeltaChangeType;
import ru.yandex.disk.search.face.DeltaItem;
import ru.yandex.disk.search.face.FaceModification;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicTypeSafeJsonParser;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.DiskBackendFields;
import ru.yandex.ps.disk.search.FaceBackendFields;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class FaceDeltasHandler implements ProxyRequestHandler {
    private static final long FAILOVER_DELAY = 300L;
    private static final String GET_FIELDS =
        FaceBackendFields.FACEDELTA_DATA.stored() + ',' +
        FaceBackendFields.FACEDELTA_CLUSTER_ID.stored() + ',' +
        FaceBackendFields.FACEDELTA_VERSION.stored();

    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private static final Set<String> RESOURCE_GET_FIELDS = Collections.emptySet();

    private static final EnumParser<DeltaChangeType> DCT_PARSER
        = new EnumParser<>(DeltaChangeType.class);

    private final Proxy proxy;
    private final String faceQueue;

    public FaceDeltasHandler(final Proxy proxy) {
        this.proxy = proxy;

        this.faceQueue = System.getProperty("FACE_QUEUE", "disk_queue");
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Context context = new Context(session);

        StringBuilder sb = new StringBuilder();
        sb.append(DiskBackendFields.TYPE.prefixed());
        sb.append(':');
        sb.append("face_delta");
        sb.append(" AND ");
        sb.append(FaceBackendFields.FACEDELTA_VERSION.prefixed());
        sb.append(":[");
        sb.append(context.version + 1);
        sb.append(" TO ");
        sb.append(Long.MAX_VALUE);
        sb.append("]");

        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());
        qc.append("service", context.user().service());
        qc.append("length", context.length());
        qc.append("sort", FaceBackendFields.FACEDELTA_VERSION.stored());
        qc.append("asc", "true");
        qc.append("text", sb.toString());
        qc.append("get", GET_FIELDS);

        session.logger().info("Hosts: " + proxy.searchMap().searchHosts(context.user));
        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new DeltasCallback(context));
    }

    private final class DeltasCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Context context;

        public DeltasCallback(
            final Context context)
            throws BadRequestException
        {
            super(context.session);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject response) {
            BasicTypeSafeJsonParser parser = new BasicTypeSafeJsonParser();
            long maxVersion = 0;

            ResourcesCallback fcb;
            List<Delta> deltas;
            Set<String> resourcesIds;
            boolean huge;
            try {
                JsonMap responseMap = response.asMap();
                int hits = responseMap.getInt("hitsCount");
                huge = hits > context.length();
                JsonList diffs = responseMap.getList("hitsArray");
                resourcesIds = new LinkedHashSet<>(diffs.size() << 1);
                deltas = new ArrayList<>(diffs.size());

                int versions = 0;
                for (JsonObject item: diffs) {
                    JsonMap diff = item.asMap();
                    String data =
                        diff.getString(FaceBackendFields.FACEDELTA_DATA.stored());
                    String clusterId = diff.getString(FaceBackendFields.FACEDELTA_CLUSTER_ID.stored());
                    long version = diff.getLong(FaceBackendFields.FACEDELTA_VERSION.stored());
                    if (version > maxVersion) {
                        maxVersion = version;
                        versions += 1;
                    }
                    JsonList actions = parser.parse(data).asList();
                    for (JsonObject actionObj: actions) {
                        JsonMap action = actionObj.asMap();
                        String deltaTypeStr = action.getString("type");
                        DeltaChangeType changeType = DCT_PARSER.apply(deltaTypeStr);

                        DeltaItem deltaItem;
                        if (changeType == DeltaChangeType.ITEM_ADDED
                            || changeType == DeltaChangeType.ITEM_DELETED)
                        {
                            FaceModification fm =
                                FaceModification.parse(action, changeType);
                            if (changeType == DeltaChangeType.ITEM_ADDED) {
                                resourcesIds.add(fm.faceId());
                            }

                            deltaItem = fm;
                        } else {
                            deltaItem = ClusterModification.parse(action, changeType);
                        }

                        deltas.add(new Delta(changeType, clusterId, version, deltaItem));
                    }
                }

                if (huge && versions <= 1) {
                    failed(new ServerException(HttpStatus.SC_REQUEST_TOO_LONG, "Too much deltas per 1 version"));
                    return;
                }

                if (huge) {
                    fcb = new ResourcesCallback(context, maxVersion - 1, maxVersion, deltas);
                } else {
                    fcb = new ResourcesCallback(context, maxVersion, maxVersion, deltas);
                }

                context.logger().warning(
                    "Deltas found " + deltas.size() + " resourceids to fetch " + resourcesIds.size());

                if (!resourcesIds.isEmpty() && !context.resourceGet().isEmpty()) {
                    QueryConstructor qc = new QueryConstructor("/search-delta-resources?");
                    qc.append("prefix", context.user().prefix().toString());
                    qc.append("service", context.user().service());
                    StringBuilder sb = new StringBuilder();
                    sb.append("id:(");
                    for (String resid: resourcesIds) {
                        int pos = resid.indexOf(':');
                        sb.append("face_");
                        if (pos > 0) {
                            sb.append(resid.replaceAll(":", "\\\\:"));
                            //sb.append(resid, pos + 1, resid.length());
                        } else {
                            sb.append(resid);
                        }
                        sb.append(" ");
                    }
                    sb.setLength(sb.length() - 1);
                    sb.append(")");

                    qc.append("text", sb.toString());

                    StringBuilder joinSb = new StringBuilder();
                    joinSb.append("left_join(");
                    joinSb.append(FaceBackendFields.FACE_RESOURCE_ID.stored());
                    joinSb.append(',');
                    joinSb.append("resource_id");
                    joinSb.append(",,");
                    for (String field: context.resourceGet()) {
                        joinSb.append(field);
                        joinSb.append(' ');
                        joinSb.append(field);
                        joinSb.append(',');
                    }
                    joinSb.setLength(joinSb.length() - 1);
                    joinSb.append(')');
                    qc.append("dp", joinSb.toString());
                    // drop all record if right part is null
                    qc.append("keep-right-null", "false");

                    StringBuilder getFields = new StringBuilder(StringUtils.join(context.resourceGet(), ','));
                    if (getFields.length() > 0) {
                        getFields.append(',');
                    }

                    getFields.append("face_coord_x,face_coord_y,face_height,face_width,face_confidence,face_gender,face_id,face_age");
                    qc.append("get", getFields.toString());
                    proxy.sequentialRequest(
                        session,
                        context,
                        new BasicAsyncRequestProducerGenerator(qc.toString()),
                        FAILOVER_DELAY,
                        true,
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        context.session().listener().adjustContextGenerator(
                            context.client().httpClientContextGenerator()),
                        fcb);
                } else {
                    fcb.completed(JsonMap.EMPTY);
                }
            } catch (JsonException | BadRequestException e) {
                failed(e);
                return;
            }
        }
    }

    private static class Delta {
        private final DeltaChangeType changeType;
        private final String clusterId;
        private final long version;
        private final DeltaItem item;
        public Delta(
            final DeltaChangeType changeType,
            final String clusterId,
            final long version,
            final DeltaItem item)
        {
            this.changeType = changeType;
            this.clusterId = clusterId;
            this.version = version;
            this.item = item;
        }

        public DeltaItem item() {
            return item;
        }


        public void writeValue(
            final JsonWriter writer,
            final Map<String, JsonMap> resources)
            throws IOException
        {
            writer.startObject();
            writer.key("type");
            writer.value(changeType.name());
            writer.key("cluster_id");
            writer.value(clusterId);
            writer.key("version");
            writer.value(version);

            if (item.face()) {
                FaceModification fm = item.asFace();
                writer.key("face_id");
                writer.value(fm.faceId());
                writer.key("resource_id");
                writer.value(fm.resourceId());
                writer.key("resource");
                writer.value((JsonValue) resources.get(fm.faceId()));
            }
            writer.endObject();
        }
    }

    private static class ResourcesCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final List<Delta> deltas;
        private final JsonType jsonType;
        private final Context context;
        private final long maxPrintVersion;
        private final long maxVersion;

        public ResourcesCallback(
            final Context context,
            final long maxPrintVersion,
            final long maxVersion,
            final List<Delta> deltas)
            throws BadRequestException
        {
            super(context.session());
            this.deltas = deltas;

            this.maxPrintVersion = maxPrintVersion;
            this.maxVersion = maxVersion;
            this.context = context;
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final JsonObject result) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = jsonType.create(sbw)) {
                Map<String, JsonMap> resMap;
                if (result != JsonMap.EMPTY) {
                    JsonList resourcesList = result.asMap().getList("hitsArray");

                    resMap = new LinkedHashMap<>(resourcesList.size() << 1);
                    for (JsonObject item: resourcesList) {
                        JsonMap map = item.asMap();
                        String faceId = map.getString(FaceBackendFields.FACE_ID.stored());
                        String rid = map.getString("resource_id");
                        if (map.getString("width", null) == null
                            && context.resourceGet().contains("width"))
                        {
                            context.logger().warning(
                                "Skipping delta, no width " + JsonType.NORMAL.toString(rid));
                            continue;
                        }

                        if (map.getString("height", null) == null
                            && context.resourceGet().contains("height"))
                        {
                            context.logger().warning(
                                "Skipping delta, no height " + JsonType.NORMAL.toString(rid));
                            continue;
                        }

                        resMap.put(faceId, map);
                    }
                } else {
                    resMap = Collections.emptyMap();
                }

                writer.startObject();
                writer.key("items");
                writer.startArray();
                long version = context.version;
                for (Delta delta: deltas) {
                    if (delta.version > maxPrintVersion) {
                        context.logger().warning(
                            "Skipping delta wrong version "
                                + delta.version + " > " + maxPrintVersion + " " + delta.clusterId);
                        continue;
                    }
                    if (delta.version > version) {
                        version = delta.version;
                    }
                    if (delta.item().face()
                        && delta.changeType == DeltaChangeType.ITEM_ADDED
                        && !context.resourceGet().isEmpty()
                        && !resMap.containsKey(delta.item().asFace().faceId()))
                    {
                        context.logger().warning(
                            "Skipping delta resource not found "
                                + delta.version + "  "
                                + delta.item().asFace().faceId() + " "
                                + delta.item().asFace().resourceId());
                        continue;
                    }

                    delta.writeValue(writer, resMap);
                }
                writer.endArray();
                writer.key("version");
                writer.value(version);
                writer.key("next_version");
                writer.value(maxVersion);
                writer.key("has_more");
                writer.value(maxVersion > version);
                writer.endObject();
            } catch (IOException | JsonException e) {
                failed(e);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(session.acceptedCharset())));
        }
    }

    private final class Context
        implements UniversalSearchProxyRequestContext
    {
        private final Long uid;
        private final User user;
        private final boolean allowLaggingHosts;

        private final ProxySession session;
        private final AsyncClient client;
        private final JsonType jsonType;
        private final long version;
        private final int length;
        private final Set<String> resourceGet;

        public Context(final ProxySession session) throws BadRequestException {
            this.session = session;
            this.version = session.params().getLong("version");
            this.length = session.params().get("length", 300, PositiveIntegerValidator.INSTANCE);
            this.uid = session.params().getLong("uid");
            this.user = new User(faceQueue, new LongPrefix(uid));
            allowLaggingHosts =
                session.params().getBoolean("allow-lagging-hosts", false);
            client = proxy.searchClient().adjust(session.context());
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            resourceGet = session.params().get("resource_get", RESOURCE_GET_FIELDS, SET_PARSER);
        }

        public int length() {
            return length;
        }

        public Set<String> resourceGet() {
            return resourceGet;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public Long uid() {
            return uid;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE : 0L;
        }
    }
}
