package ru.yandex.search.disk.proxy.ipdd;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;

import ru.yandex.collection.LongPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.HttpEntitySendingCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.AbstractUidProxyHandler;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.request.util.SearchRequestText;

public class IpddClusterizeHandler extends AbstractUidProxyHandler {
    private static final CollectionParser<String, Set<String>, Exception>
        EXCLUDE_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);
    private static final long MILLIS_PER_DAY = 86400000L;
    private static final int EVENT_TYPE_LENGTH = 128;
    private static final int MIN_COMPRESS_LENGTH = 10;
    private static final String TO = " TO ";
    private static final String POSTFILTER = "postfilter";
    private static final String DP = "dp";
    private static final String OFFSET = "offset";

    public IpddClusterizeHandler(final Proxy proxy) {
        super(proxy, proxy.ipddService());
    }

    public static StringBuilder constructFilterQuery(
        final ProxySession session,
        final QueryConstructor query)
        throws HttpException
    {
        return constructFilterQuery(
            new DaysOptimizationContext(
                session,
                Collections.emptyList(),
                Long.MAX_VALUE,
                Long.MAX_VALUE),
            query);
    }

    // CSOFF: ParameterNumber
    public static StringBuilder constructFilterQuery(
        final DaysOptimizationContext context,
        final QueryConstructor query)
        throws HttpException
    {
        CgiParams params = context.params();
        Long from = params.get(
            "end_timestamp",
            null,
            NonNegativeLongValidator.INSTANCE);
        boolean postfilterFrom = from != null;
        String platform = params.getString("platform", null);
        String folder = params.getString("parent_folder", null);
        String eventType = params.getString("event_type", null);
        int sinceDay = context.sinceDay();
        if (sinceDay != 0
            && from == null
            && platform == null
            && folder == null
            && (eventType == null || eventType.length() > EVENT_TYPE_LENGTH))
        {
            from = sinceDay * MILLIS_PER_DAY;
        }
        long version = context.version();
        StringBuilder text = new StringBuilder("version_day:[");
        if (from == null) {
            text.append('0');
        } else {
            text.append(from);
        }
        text.append(TO);
        text.append(version);
        text.append(']');

        Long to = context.to();
        if (from != null || to != null) {
            long fromInterval;
            if (from == null) {
                fromInterval = 0L;
            } else {
                fromInterval = from;
            }
            long toInterval;
            if (to == null) {
                toInterval = version;
            } else {
                toInterval = to;
            }
            text.append(" AND event_timestamp_day:[");
            text.append(fromInterval);
            text.append(TO);
            text.append(toInterval);
            text.append(']');
        }
        if (eventType != null) {
            text.append(" AND event_type:");
            text.append(eventType);
        }
        if (platform != null) {
            text.append(" AND platform:");
            text.append(platform);
        }
        if (folder != null) {
            folder = SearchRequestText.quoteEscape(folder);
            text.append(" AND (source_path:\"");
            text.append(folder);
            text.append("\" OR source_folder:\"");
            text.append(folder);
            text.append("\" OR target_path:\"");
            text.append(folder);
            text.append("\" OR target_folder:\"");
            text.append(folder);
            text.append('"');
            text.append(')');
        }
        query.append(POSTFILTER, "version <= " + version);
        if (postfilterFrom) {
            query.append(POSTFILTER, "event_timestamp >= " + from);
        }
        if (to != null) {
            query.append(POSTFILTER, "event_timestamp <= " + to);
        }
        String only = params.getString("only_resource_type", null);
        if (only == null) {
            Set<String> excludes = params.get(
                "exclude_resource_type",
                Collections.emptySet(),
                EXCLUDE_PARSER);
            for (String exclude: excludes) {
                query.append(POSTFILTER, "resource_type != " + exclude);
            }
        } else {
            query.append(POSTFILTER, "resource_type == " + only);
        }
        return text;
    }
    // CSON: ParameterNumber

    public static BasicAsyncRequestProducerGenerator daysRequest(
        final User user)
    {
        return new BasicAsyncRequestProducerGenerator(
            "/printkeys-days?field=event_timestamp_day&print-freqs&prefix="
            + user.prefix().toString()
            + "%23&user=" + user.prefix().toString());
    }

    @Override
    public void handle(
        final ProxySession session,
        final User user,
        final List<HttpHost> hosts)
        throws HttpException
    {
        AsyncClient client = proxy.searchClient().adjust(session.context());
        client.executeWithDelay(
            hosts,
            daysRequest(user),
            0L,
            DaysConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new DaysCallback(session, user, hosts));
    }

    @Override
    public String toString() {
        return "Performs disk journal actions clusterization for user";
    }

    private class DaysCallback
        extends AbstractProxySessionCallback<List<LongPair<Integer>>>
    {
        private final User user;
        private final List<HttpHost> hosts;

        DaysCallback(
            final ProxySession session,
            final User user,
            final List<HttpHost> hosts)
        {
            super(session);
            this.user = user;
            this.hosts = hosts;
        }

        @Override
        public void completed(final List<LongPair<Integer>> days) {
            try {
                int offset = session.params().get(
                    OFFSET,
                    0,
                    NonNegativeIntegerValidator.INSTANCE);
                int length = session.params().get(
                    "max_amount",
                    Integer.MAX_VALUE,
                    NonNegativeIntegerValidator.INSTANCE);
                long maxDocs = 1L;
                if (length > 0) {
                    maxDocs += offset / length;
                }
                maxDocs *= proxy.config().fatUserDocs();
                DaysOptimizationContext context = new DaysOptimizationContext(
                    session,
                    days,
                    (((long) length + offset) << 1L) + 1L,
                    maxDocs);
                QueryConstructor query = new QueryConstructor(
                    "/clusterize-ipdd?min-cluster=1&date-field=event_timestamp"
                    + "&distance=0&group=group");
                query.append("prefix", user.prefix().toString());
                StringBuilder text = constructFilterQuery(context, query);
                query.append("text", new String(text));
                query.append("length", length);
                query.append(OFFSET, offset);
                query.copy(session.params(), "get");
                query.copy(session.params(), "interval");
                query.copyIfPresent(session.params(), "hr");
                query.copyIfPresent(session.params(), "json-type");
                query.copyIfPresent(session.params(), "skip-nulls");
                query.copyIfPresent(session.params(), "asc");
                query.copyIfPresent(
                    session.params(),
                    "max_events_per_group",
                    "merged-length");
                query.copyIfPresent(
                    session.params(),
                    "count_distinct",
                    "count-distinct");
                query.append(
                    DP,
                    "increment(event_timestamp,"
                    + session.params().getInt("tz_offset")
                    + " local_timestamp)");
                query.append(DP, "cdiv(local_timestamp,86400000 local_day)");
                query.append(DP, "descending(group_key descending_group_key)");
                query.append(
                    DP,
                    "multi(descending_group_key,local_day group)");
                BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator(query.toString());
                producerGenerator.copyHeader(
                    session.request(),
                    HttpHeaders.ACCEPT_CHARSET);
                if (length >= MIN_COMPRESS_LENGTH) {
                    producerGenerator.copyHeader(
                        session.request(),
                        HttpHeaders.ACCEPT_ENCODING);
                }
                AsyncClient client =
                    proxy.searchClient().adjust(session.context());
                client.executeWithDelay(
                    hosts,
                    producerGenerator,
                    0L,
                    NByteArrayEntityAsyncConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new HttpEntitySendingCallback(session));
            } catch (Exception e) {
                failed(e);
            }
        }
    }
}

