package ru.yandex.search.disk.proxy.ipdd;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.collection.LongPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterFactory;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.AbstractUidProxyHandler;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class IpddCountersHandler extends AbstractUidProxyHandler {
    private static final int FAILOVERDELAY = 5000;
    private static final int TIMEOUT = 21000;
    private static final String FIELD = "field";

    public IpddCountersHandler(final Proxy proxy) {
        super(proxy, proxy.ipddService());
    }

    @Override
    public String toString() {
        return "Events types counting handler";
    }

    @Override
    public void handle(
        final ProxySession session,
        final User user,
        final List<HttpHost> hosts)
        throws HttpException
    {
        AsyncClient client = proxy.searchClient().adjust(session.context());
        client.executeWithDelay(
            hosts,
            IpddClusterizeHandler.daysRequest(user),
            0L,
            DaysConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new DaysCallback(session, user, hosts));
    }

    private class DaysCallback
        extends AbstractProxySessionCallback<List<LongPair<Integer>>>
    {
        private final User user;
        private final List<HttpHost> hosts;

        DaysCallback(
            final ProxySession session,
            final User user,
            final List<HttpHost> hosts)
        {
            super(session);
            this.user = user;
            this.hosts = hosts;
        }

        @Override
        public void completed(final List<LongPair<Integer>> days) {
            try {
                QueryConstructor query = new QueryConstructor(
                    "/search?dp=const(FAKE_LUCENE_NULL_FIELD+null)&"
                    + "get=field&sort=field&asc&group=field&merge_func=count");
                query.append("prefix", user.prefix().toString());
                StringBuilder text =
                    IpddClusterizeHandler.constructFilterQuery(
                        new DaysOptimizationContext(
                            session,
                            days,
                            Long.MAX_VALUE,
                            proxy.config().fatUserDocs()),
                        query);
                query.append("text", new String(text));
                String field =
                    session.params().get(FIELD, NonEmptyValidator.INSTANCE);
                query.append("dp", "fallback(" + field + ",null field)");
                AsyncClient client =
                    proxy.searchClient().adjust(session.context());
                client.execute(
                    hosts,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    System.currentTimeMillis() + TIMEOUT,
                    FAILOVERDELAY,
                    SearchResultConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new Callback(
                        session,
                        JsonTypeExtractor.NORMAL.extract(session.params())));
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class Callback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final JsonWriterFactory writerFactory;

        Callback(
            final ProxySession session,
            final JsonWriterFactory writerFactory)
        {
            super(session);
            this.writerFactory = writerFactory;
        }

        @Override
        public void completed(final SearchResult result) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            int i = 0;
            try (JsonWriter writer = writerFactory.create(sbw)) {
                writer.startObject();
                for (SearchDocument doc: result.hitsArray()) {
                    Map<String, String> attrs = doc.attrs();
                    String attr = attrs.get(FIELD);
                    if (!attr.equals("FAKE_LUCENE_NULL_FIELD")) {
                        writer.key(attr);
                        writer.value(
                            ValueUtils.asInt(attrs.get("merged_docs_count"), 0)
                            + 1);
                    }
                }
                writer.endObject();
            } catch (IOException | JsonUnexpectedTokenException e) {
                failed(
                    new BadGatewayException(
                        "Failed to parse result #" + i + ':' + ' ' + result,
                        e));
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON.withCharset(
                        session.acceptedCharset())));
        }
    }
}

