package ru.yandex.search.disk.proxy.ipdd;

import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.AbstractUidProxyHandler;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.BasicSearchRequest;
import ru.yandex.search.rules.MisspellRule;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;
import ru.yandex.util.string.StringUtils;

public class IpddSearchHandler
    extends AbstractUidProxyHandler
    implements SearchRule<SearchResult, IpddRequestParams, SearchInfo>
{
    private static final String TEXT = "text";
    private static final FieldsTermsSupplierFactory SEARCH_FIELDS =
        new FieldsTermsSupplierFactory(
            "source_path_tokenized",
            "target_path_tokenized",
            "album_title",
            "album_prev_title",
            "item_album_title",
            "child_album_title",
            "invite_login",
            "product_name_ru",
            "product_name_ua",
            "product_name_tr",
            "product_name_en");

    private final SearchRule<SearchResult, IpddRequestParams, SearchInfo> rule;

    public IpddSearchHandler(final Proxy proxy, final boolean raw) {
        super(proxy, proxy.ipddService());
        if (raw) {
            rule = this;
        } else {
            rule = new MisspellRule<>(
                this,
                proxy.misspellClient(),
                TEXT,
                x -> x.hitsCount() == 0L);
        }
    }

    @Override
    public void handle(
        final ProxySession session,
        final User user,
        final List<HttpHost> hosts)
        throws HttpException
    {
        rule.execute(
            new BasicSearchRequest<>(
                session,
                new IpddSearchService(user, hosts)));
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, IpddRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        IpddRequestParams requestParams = request.requestParams();
        QueryConstructor query = new QueryConstructor(
            IpddGroupHandler.SEARCH_URI
            + "&offset=" + requestParams.offset()
            + "&length=" + requestParams.length());
        query.append("prefix", requestParams.user().prefix().toString());
        String rawText = params.getString(TEXT, "");
        if (rule == this) {
            // This is a raw request handler
            query.append(TEXT, rawText);
        } else {
            SearchRequestText requestText = new SearchRequestText(rawText);
            String escaped = requestText.fullEscape(true);
            StringBuilder text = IpddClusterizeHandler.constructFilterQuery(
                request.session(),
                query);
            text.append(" AND (source_path_lowercase_keyword:*");
            text.append(escaped);
            text.append("* OR target_path_lowercase_keyword:*");
            text.append(escaped);
            text.append('*');
            if (requestText.hasWords()) {
                text.append(" OR ");
                requestText.fieldsQuery(text, SEARCH_FIELDS);
            }
            text.append(')');
            requestText.negationsQuery(text, SEARCH_FIELDS);
            query.append(TEXT, new String(text));
        }
        query.append("get", StringUtils.join(requestParams.fields(), ','));
        query.copyIfPresent(params, "skip-nulls");
        AsyncClient client = proxy.searchClient().adjust(
            request.session().context());
        client.execute(
            requestParams.hosts(),
            new BasicAsyncRequestProducerGenerator(query.toString()),
            SearchResultConsumerFactory.OK,
            request.session().listener().createContextGeneratorFor(client),
            request.callback());
    }

    @Override
    public String toString() {
        return "Events search handler";
    }
}

