package ru.yandex.search.disk.proxy.querylanguage;

import java.text.ParseException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.Locale;

import ru.yandex.parser.query.AndQuery;
import ru.yandex.parser.query.FieldQuery;
import ru.yandex.parser.query.NotQuery;
import ru.yandex.parser.query.OrQuery;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryToken;
import ru.yandex.parser.query.QueryVisitor;
import ru.yandex.parser.query.QuotedQuery;
import ru.yandex.search.request.util.SearchRequestText;


public class DiskLuceneQueryConstructor extends AbstractLuceneQueryConstructor {
    private final Deque<Collection<String>> scopes = new ArrayDeque<>();

    public DiskLuceneQueryConstructor(
        final LuceneQueryContext context)
    {
        super(context);
        scopes.push(context.defaultScope());
    }

    public String request() {
        return new String(request);
    }

    private List<String> luceneFieldsNames(final String field)
        throws ParseException
    {
        List<String> luceneFields;
        switch (field.toLowerCase(Locale.ROOT)) {
            case "name":
            case "имя":
            case "название":
                luceneFields = Arrays.asList("name");
                break;
            default:
                luceneFields = null;
                break;
        }
        return luceneFields;
    }

    @Override
    public Void visit(final FieldQuery query) throws ParseException {
        List<String> fields = query.fields();
        QueryAtom subquery = query.query();
        if (fields.size() == 1) {
            boolean processed = true;
            switch (fields.get(0)) {
                case "документ":
                case "document":
                    context.session().logger().info("Subquery " + subquery);
                    new DocumentQueryConstructor().visit(subquery);
                    break;
                case "папка":
                case "folder":
                    context.session().logger().info("Subquery " + subquery);
                    new FolderQueryConstructor().visit(subquery);
                    break;
                case "вжух":
                case "magic":
                    context.session().logger().info("Magic hack");
                    context.setUseHnsw(true);
                    break;
                default:
                    processed = false;
                    break;
            }
            if (processed) {
                context.nonTrivial();
                return null;
            }
        }

        List<String> luceneFields = new ArrayList<>(fields.size() << 1);
        int unknownField = -1;
        for (int i = 0; i < fields.size(); ++i) {
            List<String> currentLuceneFields =
                luceneFieldsNames(fields.get(i));
            if (currentLuceneFields == null) {
                unknownField = i;
            } else {
                luceneFields.addAll(currentLuceneFields);
            }
        }

        if (unknownField == -1) {
            context.nonTrivial();
            scopes.push(luceneFields);
            visit(subquery);
            scopes.pop();
        } else if (subquery instanceof QueryToken) {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < fields.size(); ++i) {
                if (i != 0) {
                    sb.append(',');
                }
                sb.append(fields.get(i));
            }
            sb.append(':');
            sb.append(((QueryToken) subquery).text());
            processQueryToken(new String(sb));
        } else {
            throw new ParseException(
                "Unknown field: " + fields.get(unknownField),
                unknownField);
        }
        return null;
    }

    @Override
    public Void visit(final QuotedQuery query) throws ParseException {
//        Collection<String> scope = scopes.peek();
//        boolean empty = true;
//        request.append('(');
//        for (String field: scope) {
//            if (empty) {
//                empty = false;
//            } else {
//                request.append(OR);
//            }
//            request.append(field);
//            request.append(':');
//            request.append('"');
//            boolean first = true;
//            for (QueryToken token: query.tokens()) {
//                if (first) {
//                    first = false;
//                } else {
//                    request.append(' ');
//                }
//                request.append(
//                    token.text().replace('"', ' ').replace('\\', ' '));
//            }
//            request.append('"');
//        }
//        request.append(')');
        return null;
    }

    private void processQueryToken(final String token) throws ParseException {
        request.append(token.toLowerCase(Locale.ROOT));
//        if (token.chars().anyMatch(Character::isLetterOrDigit)) {
//            String text = token.toLowerCase(Locale.ROOT);
//            Collection<String> scope = scopes.peek();
//            boolean empty = true;
//            request.append('(');
//            text = SearchRequestText.fullEscape(text, false);
////            for (String field: scope) {
////                if (empty) {
////                    empty = false;
////                } else {
////                    request.append(OR);
////                }
////                request.append(field);
////                request.append(':');
////                request.append(text);
////            }
//            request.append(')');
//        } else {
//            request.append(context.selectAll());
//        }
    }

    @Override
    public Void visit(final QueryToken query) throws ParseException {
        processQueryToken(query.text());
        return null;
    }

    private static String join(final List<QueryToken> tokens) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < tokens.size(); ++i) {
            if (i != 0) {
                sb.append(' ');
            }
            sb.append(tokens.get(i).text());
        }
        return new String(sb);
    }

    private abstract class SpecialQueryConstructor
        extends AbstractLuceneQueryConstructor
    {
        SpecialQueryConstructor(final LuceneQueryContext context) {
            super(context);
        }

        protected abstract void process(final String text)
            throws ParseException;

        @Override
        public Void visit(final AndQuery query) throws ParseException {
            StringBuilder sb = new StringBuilder();
            ConcatQueryVisitor visitor = new ConcatQueryVisitor(sb);
            query.lhs().accept(visitor);
            sb.append(' ');
            query.rhs().accept(visitor);
            process(sb.toString());
            return null;
        }

        @Override
        public Void visit(final FieldQuery query) throws ParseException {
            DiskLuceneQueryConstructor.this.visit(query);
            return null;
        }

        @Override
        public Void visit(final QueryToken query) throws ParseException {
            process(query.text());
            return null;
        }

        @Override
        public Void visit(final QuotedQuery query) throws ParseException {
            process(join(query.tokens()));
            return null;
        }
    }

    private static class ConcatQueryVisitor implements QueryVisitor<Void, ParseException> {
        private final StringBuilder sb;

        public ConcatQueryVisitor(final StringBuilder sb) {
            this.sb = sb;
        }

        @Override
        public Void visit(AndQuery query) throws ParseException {
            visit(query.lhs());
            visit(query.rhs());
            return null;
        }

        @Override
        public Void visit(OrQuery query) throws ParseException {
            visit(query.lhs());
            visit(query.rhs());
            return null;
        }

        @Override
        public Void visit(NotQuery query) throws ParseException {
            visit(query.query());
            return null;
        }

        @Override
        public Void visit(FieldQuery query) throws ParseException {
            if (sb.length() > 0) {
                sb.append(' ');
            }
            for (String field: query.fields()) {
                sb.append(field);
                sb.append(':');
                visit(query.query());
            }
            return null;
        }

        @Override
        public Void visit(QuotedQuery query) throws ParseException {
            if (sb.length() > 0) {
                sb.append(' ');
            }

            sb.append('\"');
            boolean first = true;
            for (QueryToken token: query.tokens()) {
                if (first) {
                    first = false;
                } else {
                    sb.append(' ');
                }
                sb.append(token.text());
            }
            sb.append('\"');
            return null;
        }

        @Override
        public Void visit(QueryToken query) throws ParseException {
            sb.append(query.text());
            return null;
        }
    }

    private class DocumentQueryConstructor extends SpecialQueryConstructor {
        DocumentQueryConstructor() {
            super(DiskLuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            text = text.toLowerCase(Locale.ROOT);
            PersonalDocument docType = null;
            switch (text) {
                case "паспорт рф":
                case "internal passport":
                    docType = PersonalDocument.INTERNAL_PASSPORT;
                    break;
                case "загран":
                case "загранпаспорт":
                case "заграничный паспорт":
                    docType = PersonalDocument.INTERNATIONAL_PASSPORT;
                    break;
                case "омс":
                case "полис омс":
                    docType = PersonalDocument.OMS;
                    break;
                case "снилс":
                    docType = PersonalDocument.SNILS;
                    break;
                case "инн":
                    docType = PersonalDocument.INN;
                    break;
                case "водительское удостоверение":
                case "ву":
                    docType = PersonalDocument.DRIVER_LICENSE;
                    break;

                default:
                    break;
            }
            if (docType == null) {
                throw new ParseException("Unknown doc type: " + text, 0);
            }

            context.personalDocument(docType, text);
            request.append("type:file");
        }
    }

    private class FolderQueryConstructor extends SpecialQueryConstructor {
        public FolderQueryConstructor() {
            super(DiskLuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            //text = text.toLowerCase(Locale.ROOT);
            request.append("key:/disk/");
            request.append(SearchRequestText.fullEscape("/disk/" + context.folder(), true));
            request.append("*");
            context.folder(text);
        }
    }
}

