package ru.yandex.search.disk.proxy.rules;

import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.disk.proxy.DiskRequestParams;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class DssmSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private static final String TEXT = "text";
    private static final String DP = "dp";

    private final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next;
    private final JniWrapper dssm;
    private final String dssmPostfilter;

    public DssmSearchRule(
        final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next,
        final JniWrapper dssm,
        final long dssmThreshold)
    {
        this.next = next;
        this.dssm = dssm;
        dssmPostfilter = buildDssmPostfilter(dssmThreshold);
    }

    public static String buildDssmPostfilter(final long dssmThreshold) {
        return "filter_cmp(i2tscore,>=," + dssmThreshold + ')';
    }

    public static String applyDssmOrNull(
        final ProxySession session,
        final JniWrapper dssm,
        final String text)
    {
        if (!text.isEmpty()) {
            String result;
            try {
                result = dssm.apply(text, null).process(null, 0, 0);
                session.logger().info("Created dssm for " + text);
            } catch (JniWrapperException e) {
                session.logger().log(
                    Level.WARNING,
                    "Failed to execute dssm on text '" + text + '\'',
                    e);
                return null;
            }
            return result;
        } else {
            return text;
        }
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        CgiParams cgiParams = request.cgiParams();
        String text = cgiParams.getString(TEXT, "");
        String dssmText = applyDssmOrNull(request.session(), dssm, text);
        if (dssmText == null) {
            text = "";
        } else {
            text = dssmText;
        }

        if (text.isEmpty()) {
            request.callback().completed(SearchResult.EMPTY);
            return;
        }
        cgiParams = new CgiParams(cgiParams);
        cgiParams.replace(TEXT, "has_i2t:1");
        cgiParams.replace(
            DP,
            "chex-dot-product(i2t_keyword " + text + " i2tscore)");

        Long customDssmTh = cgiParams.getLong("dssm_threshold", null);
        String dssmPostfilter = this.dssmPostfilter;
        if (customDssmTh != null) {
            dssmPostfilter = buildDssmPostfilter(customDssmTh);
        }
        cgiParams.add(DP, dssmPostfilter);
        cgiParams.replace("how", "i2tscore");
        next.execute(request.withCgiParams(cgiParams));
    }
}

