package ru.yandex.search.disk.proxy.rules;

import java.util.Arrays;
import java.util.Collections;
import java.util.Locale;
import java.util.function.Function;
import java.util.function.Supplier;

import org.apache.http.HttpException;

import ru.yandex.function.ConcatSupplierFactory;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.disk.proxy.DiskRequestParams;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.QueryAppender;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class FileSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private static final String TEXT = "text";

    private final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next;
    private final Function<
        ? super String,
        ? extends Supplier<? extends QueryAppender>>
        queriesFactory;
    private final Function<
        ? super String,
        ? extends Supplier<? extends QueryAppender>>
        foldersQueriesFactory;

    public FileSearchRule(
        final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next,
        final Function<
            ? super String,
            ? extends Supplier<? extends QueryAppender>>
        queriesFactory)
    {
        this.next = next;
        this.queriesFactory = queriesFactory;
        foldersQueriesFactory =
            new ConcatSupplierFactory<>(
                Arrays.asList(
                    new FieldsTermsSupplierFactory(
                        Collections.singletonList("folder_tokenized")),
                    queriesFactory));
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        CgiParams cgiParams = new CgiParams(request.cgiParams());
        String srcText =
            new SearchRequestText.SuggestWordsModifier(
                cgiParams.getLocale("locale", Locale.ROOT))
                .apply(cgiParams.getString(TEXT, ""));
        SearchRequestText text = new SearchRequestText(srcText);
        StringBuilder sb = new StringBuilder("type:file ");
        if (text.hasWords()) {
            sb.append("AND ");
            text.fieldsQuery(sb, foldersQueriesFactory);
            sb.append(" AND ");
            text.fieldsQuery(sb, queriesFactory, ") OR (");
        }
        text.negationsQuery(sb, foldersQueriesFactory);
        cgiParams.replace(TEXT, new String(sb));
        next.execute(request.withCgiParams(cgiParams));
    }
}

