package ru.yandex.search.disk.proxy.rules;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.CollectionsComparator;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.disk.proxy.DiskRequestParams;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.FilterSearchDocument;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;
import ru.yandex.util.string.StringUtils;

public class FoldersSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private static final CollectionsComparator<String> PATH_COMPARATOR =
        CollectionsComparator.naturalOrder();

    private static final Comparator<Folder> NAME_COMPARATOR =
        new Comparator<Folder>() {
            @Override
            public int compare(final Folder lhs, final Folder rhs) {
                int cmp = lhs.name.compareTo(rhs.name);
                if (cmp == 0) {
                    cmp = PATH_COMPARATOR.compare(lhs.path, rhs.path);
                }
                return cmp;
            }
        };

    private static final String TEXT = "text";
    private static final FieldsTermsSupplierFactory FOLDER_TERMS_FACTORY =
        new FieldsTermsSupplierFactory(
            Collections.singletonList("folder_tokenized"));

    private final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next;

    public FoldersSearchRule(
        final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next)
    {
        this.next = next;
    }

    private static List<Folder> fold(final List<SearchDocument> docs) {
        List<Folder> folders = new ArrayList<>(docs.size());
        for (SearchDocument doc: docs) {
            Folder folder = new Folder(doc);
            if (folder.valid()) {
                folders.add(new Folder(doc));
            }
        }
        if (folders.isEmpty()) {
            return folders;
        }
        Collections.sort(folders);
        return foldFolders(folders);
    }

    private static List<Folder> foldFolders(final List<Folder> folders) {
        List<Folder> result = new ArrayList<>();
        if (!folders.isEmpty()) {
            Folder prev = folders.get(0);
            result.add(prev);
            for (int i = 1; i < folders.size(); ++i) {
                Folder folder = folders.get(i);
                if (!folder.startsWith(prev)) {
                    result.add(folder);
                    prev = folder;
                }
            }
            Collections.sort(result, NAME_COMPARATOR);
        }
        return result;
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        CgiParams cgiParams = new CgiParams(request.cgiParams());
        cgiParams.replace(
            "get",
            StringUtils.join(
                request.requestParams().fields(),
                ',',
                "id,name,folder,",
                ""));
        cgiParams.replace("collector", "passthru(0)");
        cgiParams.remove("how");
        cgiParams.replace("length", Integer.toString(Integer.MAX_VALUE));
        String srcText =
            new SearchRequestText.SuggestWordsModifier(
                cgiParams.getLocale("locale", Locale.ROOT))
                .apply(cgiParams.getString(TEXT, ""));
        if (srcText.isEmpty()) {
            cgiParams.replace(TEXT, "type:dir");
            next.execute(
                request
                    .withCgiParams(cgiParams)
                    .withCallback(new FoldingCallback(request.callback())));
        } else {
            SearchRequestText text = new SearchRequestText(srcText);
            FutureCallback<SearchResult> keywordCallback;
            if (!text.isEmpty()) {
                DoubleFutureCallback<SearchResult, SearchResult> callback =
                    new DoubleFutureCallback<>(
                        new FoldersCallback(request.callback()));
                keywordCallback = callback.second();
                StringBuilder sb = new StringBuilder("type:dir ");
                if (text.hasWords()) {
                    sb.append("AND ");
                    text.fieldsQuery(sb, FOLDER_TERMS_FACTORY);
                }
                text.negationsQuery(sb, FOLDER_TERMS_FACTORY);
                cgiParams.replace(TEXT, new String(sb));
                next.execute(
                    request
                        .withCgiParams(cgiParams)
                        .withCallback(callback.first()));
            } else {
                keywordCallback = new FoldingCallback(request.callback());
            }

            // workaround:now backend could do 400 on request with *xxxx*
            // we should not fail entire request
            FutureCallback<? super SearchResult> leadingWildcardCallback =
                new ErrorSuppressingFutureCallback<>(
                    keywordCallback,
                    (e) -> RequestErrorType.IO,
                    SearchResult.EMPTY);
            CgiParams keywordParams = new CgiParams(cgiParams);
            keywordParams.replace(
                TEXT,
                "type:dir AND folder:*" + text.fullEscape(true) + '*');
            next.execute(
                request
                    .withCgiParams(keywordParams)
                    .withCallback(leadingWildcardCallback));
        }
    }

    private static class FoldingCallback
        extends AbstractFilterFutureCallback<SearchResult, SearchResult>
    {
        FoldingCallback(final FutureCallback<? super SearchResult> callback) {
            super(callback);
        }

        @Override
        public void completed(final SearchResult result) {
            callback.completed(
                new BasicSearchResult(
                    new ArrayList<SearchDocument>(fold(result.hitsArray())),
                    result.hitsCount()));
        }
    }

    private static class FoldersCallback
        extends AbstractFilterFutureCallback<
            Map.Entry<SearchResult, SearchResult>,
            SearchResult>
    {
        FoldersCallback(final FutureCallback<? super SearchResult> callback) {
            super(callback);
        }

        @Override
        public void completed(
            final Map.Entry<SearchResult, SearchResult> result)
        {
            List<Folder> tokenized = fold(result.getKey().hitsArray());
            List<Folder> keyword = fold(result.getValue().hitsArray());
            List<Folder> all = new ArrayList<>(tokenized);
            all.addAll(keyword);
            all = foldFolders(all);
            tokenized.retainAll(all);
            all.removeAll(tokenized);
            keyword.retainAll(all);
            tokenized.addAll(keyword);
            callback.completed(
                new BasicSearchResult(
                    new ArrayList<SearchDocument>(tokenized),
                    result.getKey().hitsCount()
                    + result.getValue().hitsCount()));
        }
    }

    private static class Folder
        extends FilterSearchDocument
        implements Comparable<Folder>
    {
        private final String id;
        private final String name;
        private final List<String> path;

        Folder(final SearchDocument document) {
            super(document);
            id = attrs().get("id");
            name = attrs().get("name");
            String folder = attrs().get("folder");
            if (folder == null) {
                path = null;
            } else {
                // TODO: Implement non-trimming ListParser and use here
                path = Arrays.asList(folder.split("/"));
            }
        }

        public boolean valid() {
            return id != null && name != null && path != null;
        }

        public boolean startsWith(final Folder prefix) {
            return CollectionsComparator.startsWith(path, prefix.path);
        }

        @Override
        public int compareTo(final Folder other) {
            return PATH_COMPARATOR.compare(path, other.path);
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof Folder && id.equals(((Folder) o).id);
        }

        @Override
        public int hashCode() {
            return id.hashCode();
        }
    }
}

