package ru.yandex.search.disk.proxy.rules;

import org.apache.http.HttpException;

import ru.yandex.geocoder.GeocoderRequest;
import ru.yandex.geocoder.GeocoderResult;
import ru.yandex.geocoder.GeocoderResultBuilder;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.disk.proxy.DiskRequestParams;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class GeoSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private static final String TEXT = "text";
    private static final String TO = " TO ";
    private static final GeocoderResult EMPTY_RESULT =
        new GeocoderResultBuilder().build();

    private final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next;
    private final Proxy proxy;

    public GeoSearchRule(
        final Proxy proxy,
        final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next)
    {
        this.proxy = proxy;
        this.next = next;
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        String text = request.cgiParams().getString(TEXT, "").trim();
        if (text.isEmpty()) {
            request.callback().completed(SearchResult.EMPTY);
        } else {
            GeocoderRequest geocoderRequest = new GeocoderRequest(
                proxy.config().geoSearchConfig());
            geocoderRequest.addHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                proxy.geoTvm2Ticket());
            geocoderRequest.setText(text);
            proxy.geoSearchClient().execute(
                geocoderRequest,
                request.session().listener().createContextGeneratorFor(
                    proxy.geoSearchClient()),
                new ErrorSuppressingFutureCallback<>(
                    new Callback(request, next),
                    EMPTY_RESULT));
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<GeocoderResult, SearchResult>
    {
        private final SearchRequest<
            SearchResult,
            DiskRequestParams,
            SearchInfo>
            request;
        private final SearchRule<SearchResult, DiskRequestParams, SearchInfo>
            next;

        Callback(
            final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
            request,
            final SearchRule<SearchResult, DiskRequestParams, SearchInfo> next)
        {
            super(request.callback());
            this.request = request;
            this.next = next;
        }

        @Override
        public void completed(final GeocoderResult result) {
            int size = result.size();
            if (size == 0) {
                callback.completed(SearchResult.EMPTY);
            } else {
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < size; ++i) {
                    if (i != 0) {
                        sb.append(" OR ");
                    }
                    sb.append("(latitude:[");
                    sb.append(result.lowerLatitude(i));
                    sb.append(TO);
                    sb.append(result.upperLatitude(i));
                    sb.append("] AND longitude:[");
                    sb.append(result.lowerLongitude(i));
                    sb.append(TO);
                    sb.append(result.upperLongitude(i));
                    sb.append("])");
                }
                CgiParams cgiParams = new CgiParams(request.cgiParams());
                cgiParams.replace(TEXT, new String(sb));
                try {
                    next.execute(request.withCgiParams(cgiParams));
                } catch (HttpException e) {
                    failed(e);
                }
            }
        }
    }
}

