package ru.yandex.search.disk.proxy.rules;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import org.apache.http.HttpException;

import ru.yandex.collection.LongPair;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.TimingFutureCallback;
import ru.yandex.search.disk.proxy.DiskRequestParams;
import ru.yandex.search.disk.proxy.MultiSearchStat;
import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class MultiSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private final NamedRule[] rules;
    private final Consumer<? super MultiSearchStat> stater;

    public MultiSearchRule(
        final List<NamedRule> rules,
        final Consumer<? super MultiSearchStat> stater)
    {
        this.rules = rules.toArray(new NamedRule[rules.size()]);
        this.stater = stater;
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        MultiFutureCallback<LongPair<Map.Entry<String, SearchResult>>>
            callback = new MultiFutureCallback<>(
                new Callback(
                    request,
                    stater,
                    request.requestParams().offset()
                    + request.requestParams().length()));
        for (int i = 0; i < rules.length; ++i) {
            NamedRule rule = rules[i];
            rule.execute(
                request.withCallback(
                    new PayloadFutureCallback<>(
                        rule.name(),
                        new TimingFutureCallback<>(
                            callback.newCallback()))));
        }
        callback.done();
    }

    private static class Callback
        extends AbstractFilterFutureCallback<
            List<LongPair<Map.Entry<String, SearchResult>>>,
            SearchResult>
    {
        private final long start = System.currentTimeMillis();
        private final int maxDocs;
        private final Consumer<? super MultiSearchStat> stater;

        Callback(
            final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
            request,
            final Consumer<? super MultiSearchStat> stater,
            final int maxDocs)
        {
            super(request.callback());
            this.stater = stater;
            this.maxDocs = maxDocs;
        }

        @Override
        public void completed(
            final List<LongPair<Map.Entry<String, SearchResult>>> results)
        {
            long hitsCount = 0L;
            for (LongPair<Map.Entry<String, SearchResult>> result: results) {
                hitsCount += result.second().getValue().hitsCount();
            }
            List<SearchDocument> docs = new ArrayList<>(maxDocs);
            if (maxDocs > 0) {
                long totalTime = System.currentTimeMillis() - start;
                Map<String, Long> timings = new HashMap<>(results.size() << 1);
                Set<String> ids = new HashSet<>(maxDocs << 1);
                for (LongPair<Map.Entry<String, SearchResult>> result
                    : results)
                {
                    String ruleName = result.second().getKey();
                    timings.put(ruleName, result.first());
                    for (SearchDocument document
                            : result.second().getValue().hitsArray())
                    {
                        if (ids.add(document.attrs().get("id"))) {
                            docs.add(document);
                            if (docs.size() >= maxDocs) {
                                break;
                            }
                        }
                    }
                    if (docs.size() >= maxDocs) {
                        break;
                    }
                }
                stater.accept(new MultiSearchStat(totalTime, timings));
            }
            callback.completed(new BasicSearchResult(docs, hitsCount));
        }
    }
}

