package ru.yandex.search.disk.proxy.rules;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.disk.search.DiskParams;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.DiskRequestParams;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;
import ru.yandex.util.string.StringUtils;

public class PlainSearchRule
    implements SearchRule<SearchResult, DiskRequestParams, SearchInfo>
{
    private static final String GET = "get";
    private static final String LENGTH = "length";
    private static final String TEXT = "text";

    protected final Proxy proxy;
    private final Long failoverSearchDelay;
    private final int seed;

    public PlainSearchRule(final Proxy proxy, final int seed) {
        this.proxy = proxy;
        this.seed = seed;
        this.failoverSearchDelay = proxy.config().failoverSearchDelay();
    }

    private static String addCondition(
        final String text,
        final String condition)
    {
        if (text == null) {
            return condition;
        } else {
            return StringUtils.concat('(', text, ") AND ", condition);
        }
    }

    public static String defaultGet(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
    {
        Collection<String> fields = request.requestParams().fields();
        return StringUtils.join(fields, ',', "id,", "");
    }

    protected String baseUri() {
        proxy.hnswCalls().accept(0L);
        return "/search-search";
    }

    protected String getUri(boolean ignored) {
        return this.baseUri();
    }

    @Override
    public void execute(
        final SearchRequest<SearchResult, DiskRequestParams, SearchInfo>
        request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        String text = params.getString(TEXT, "");
        if (text.isEmpty()) {
            text = null;
        }

        Long failoverSearchDelay = params.getLong("search_failover", this.failoverSearchDelay);

        String visible = params.getString("visible", "");
        if (!visible.isEmpty()) {
            text = addCondition(
                text,
                "visible:\"" + SearchRequestText.quoteEscape(visible) + '"');
        }
        AuxFolders auxFolders = new AuxFolders(params);
        if (!auxFolders.isEmpty()) {
            StringBuilder sb = new StringBuilder("aux_folder:(");
            boolean first = true;
            for (String auxFolder: auxFolders) {
                if (first) {
                    first = false;
                } else {
                    sb.append(' ');
                }
                sb.append('"');
                sb.append(SearchRequestText.quoteEscape(auxFolder));
                sb.append('"');
            }
            sb.append(')');
            text = addCondition(text, new String(sb));
        }
        String key = auxFolders.key();
        if (key != null && !auxFolders.simpleKey()) {
            boolean asterisk = key.charAt(key.length() - 1) == '*';
            if (asterisk) {
                key = key.substring(0, key.length() - 1);
            }
            key = SearchRequestText.fullEscape(
                key,
                !key.isEmpty() && key.charAt(0) == '*');
            if (asterisk) {
                key = key + '*';
            }
            text = addCondition(text, "key:" + key);
        }

        if (!request.requestParams().fileExtensions().isEmpty()) {
            text = addCondition(text, StringUtils.concat("ext:(",StringUtils.join(request.requestParams().fileExtensions(), ' '), ")"));
        }
        String prefix = request.requestParams().user().prefix().toString();
        if (text == null) {
            text = "type:*";
        }
        text = '(' + text + ')' + ' ' + proxy.searchPostFilter();

        String uri = getUri(params.get("use_hnsw") != null);

        QueryConstructor query =
            new QueryConstructor(uri + "?json-type=dollar");
        query.append("prefix", prefix);
        query.append("update-prefix-activity", "true");
        if (request.requestParams().asc()) {
            query.sb().append("&asc");
        }
        String get = params.getString(GET, null);
        if (get == null) {
            get = defaultGet(request);
        }
        query.append(GET, get);
        query.append(TEXT, text);
        query.copyIfPresent(params, "collector");
        query.copyIfPresent(params, "how", "sort");
        query.copyAll(params, "dp");
        if (params.getBoolean(DiskParams.FAST_MOVED, false)) {
            Proxy.addFastMovedDp(query);
        }
        query.copyAll(params, "postfilter");
        String length = params.getString(LENGTH, null);
        if (length == null) {
            length = Integer.toString(
                request.requestParams().offset()
                + request.requestParams().length());
        }
        query.append(LENGTH, length);
        AsyncClient client =
            proxy.searchClient().adjust(request.session().context());
        List<HttpHost> hosts = request.requestParams().hosts();
        if (hosts.size() == 1) {
            request.session().subscribeForCancellation(
                client.execute(
                    hosts,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    SearchResultConsumerFactory.OK,
                    request.session().listener().createContextGeneratorFor(
                        client),
                    request.callback()));
        } else {
            hosts = new ArrayList<>(hosts);
            Collections.rotate(hosts, seed);
            if (failoverSearchDelay == null) {
                request.session().subscribeForCancellation(
                    client.execute(
                        hosts,
                        new BasicAsyncRequestProducerGenerator(query.toString()),
                        SearchResultConsumerFactory.OK,
                        request.session().listener().createContextGeneratorFor(
                            client),
                        request.callback()));
            } else {
                request.session().subscribeForCancellation(
                    client.executeWithDelay(
                        hosts,
                        new BasicAsyncRequestProducerGenerator(query.toString()),
                        failoverSearchDelay,
                        SearchResultConsumerFactory.OK,
                        request.session().listener().createContextGeneratorFor(
                            client),
                        request.callback()));
            }
        }
    }
}

