package ru.yandex.search.disk.proxy.suggest;

import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.disk.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class BasicSuggestRequestContext
    implements RequestProvider,
        SuggestRequestContext,
        SuggestRequestContextProvider
{
    private static final CollectionParser<
        SuggestType,
        Set<SuggestType>,
        ? extends Exception>
        TYPES_PARSER = new CollectionParser<>(
            new EnumParser<>(SuggestType.class),
            () -> EnumSet.noneOf(SuggestType.class));
    private static final CollectionParser<String, Set<String>, Exception>
        EXTENSIONS_PARSER = new CollectionParser<>(
        NonEmptyValidator.INSTANCE,
        LinkedHashSet::new);

    private final Proxy proxy;
    private final ProxySession session;
    private final Set<SuggestType> types;
    private final int length;
    private final Long timeout;
    private final JsonType jsonType;
    private final User user;
    private final AsyncClient client;
    private final Supplier<? extends HttpClientContext> contextGenerator;
    private final String request;
    private final boolean allowLaggingHosts;
    private final Set<String> fileExtensions;

    public BasicSuggestRequestContext(
        final Proxy proxy,
        final ProxySession session,
        final String serviceName)
        throws BadRequestException
    {
        this.proxy = proxy;
        this.session = session;
        CgiParams params = session.params();
        types = params.get("suggest-types", TYPES_PARSER);
        length = params.get("length", PositiveIntegerValidator.INSTANCE);
        timeout = params.get("timeout", null, DurationParser.POSITIVE_LONG);
        jsonType = JsonTypeExtractor.NORMAL.extract(params);
        user = new User(
            serviceName,
            params.get("uid", proxy.searchMap().prefixType(serviceName)));
        client = proxy.searchClient().adjust(session.context());
        contextGenerator =
            session.listener().createContextGeneratorFor(client);
        request = params.getString("request", "");
        allowLaggingHosts = params.getBoolean("allow-lagging-hosts", false);
        fileExtensions =
            session.params().get(
                "extension",
                Collections.emptySet(),
                EXTENSIONS_PARSER);
    }

    @Override
    public Proxy proxy() {
        return proxy;
    }

    @Override
    public ProxySession session() {
        return session;
    }

    @Override
    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }

    @Override
    public Set<SuggestType> types() {
        return types;
    }

    @Override
    public int length() {
        return length;
    }

    @Override
    public Long timeout() {
        return timeout;
    }

    @Override
    public JsonType jsonType() {
        return jsonType;
    }

    // UniversalSearchProxyRequestContext implementation
    @Override
    public User user() {
        return user;
    }

    @Override
    public Long minPos() {
        return null;
    }

    @Override
    public AsyncClient client() {
        return client;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    // RequestProvider implementation
    @Override
    public String request() {
        return request;
    }

    // SuggestRequestContextProvider implementation
    @Override
    public SuggestRequestContext suggestRequestContext() {
        return this;
    }

    @Override
    public long lagTolerance() {
        return allowLaggingHosts ? Long.MAX_VALUE : 0L;
    }

    @Override
    public Set<String> fileExtensions() {
        return fileExtensions;
    }
}

