package ru.yandex.search.disk.proxy.suggest;

import java.io.IOException;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class PromoSuggestHandler implements ProxyRequestHandler {
    private final Proxy proxy;
    private final String serviceName;
    private final PrefixType prefixType;

    public PromoSuggestHandler(final Proxy proxy) {
        this.proxy = proxy;
        serviceName = proxy.diskService();
        prefixType = proxy.searchMap().prefixType(serviceName);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        Prefix prefix = session.params().get("uid", prefixType);
        AsyncClient client = proxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                new User(serviceName, prefix),
                null,
                true,
                client,
                session.logger());
        QueryConstructor query = new QueryConstructor(
            "/search?json-type=dollar&IO_PRIO=0&sort=mtime&length=1&get=name"
            + "&text=type:file+OR+type:dir");
        query.append("prefix", prefix.toString());
        query.append("service", serviceName);
        proxy.parallelRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new Callback(
                session,
                JsonTypeExtractor.NORMAL.extract(session.params())));
    }

    @Override
    public String toString() {
        return "Suggest user to enter something in search field: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "yadisk-search-proxy/suggest/#promo";
    }

    private static class Callback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final JsonType jsonType;

        Callback(final ProxySession session, final JsonType jsonType) {
            super(session);
            this.jsonType = jsonType;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList docs = result.get("hitsArray").asList();
                String name;
                if (docs.isEmpty()) {
                    name = null;
                } else {
                    name = docs.get(0).get("name").asStringOrNull();
                }
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = jsonType.create(sbw)) {
                    writer.startObject();
                    writer.key("promo-suggest");
                    writer.value(name);
                    writer.endObject();
                }
                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(session.acceptedCharset())));
            } catch (IOException | JsonException e) {
                failed(e);
            }
        }
    }
}

