package ru.yandex.search.disk.proxy.suggest;

import java.util.Collections;
import java.util.EnumMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.search.disk.proxy.PrintkeysWarmable;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.disk.proxy.suggest.rules.FilesSuggestRule;
import ru.yandex.search.disk.proxy.suggest.rules.HistorySuggestRule;
import ru.yandex.search.disk.proxy.suggest.rules.providers.SuggestRequestContextRequestProvider;
import ru.yandex.search.disk.proxy.suggest.rules.providers.SuggestRequestContextRequestsProvider;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.ConcatResultsRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.TranslitRule;

public class SuggestHandler implements ProxyRequestHandler, PrintkeysWarmable {
    private static final long TIMEOUT_OVERHEAD = 10L;
    private static final long MIN_TIMEOUT = 50L;
    private static final Supplier<List<SuggestItem>> NULL_SUPPLIER =
        () -> null;
    private static final String SUGGEST = " suggest";

    private final Timer timer = new Timer("Suggest-Timer", true);
    private final Map<
        SuggestType,
        SearchRule<BasicSuggestRequestContext, List<SuggestItem>>> rules =
            new EnumMap<>(SuggestType.class);
    private final Proxy proxy;
    private final String serviceName;
    private final Set<String> warmFields;

    public SuggestHandler(final Proxy proxy) {
        this.proxy = proxy;
        serviceName = proxy.diskService();
        warmFields = new LinkedHashSet<>();
        // TODO: wrap every rule with TranslitRule and ConcatResultsRule
        rules.put(
            SuggestType.HISTORY,
            new TranslitRule<>(
                new ChainedSearchRule<>(
                    new ConcatResultsRule<>(
                        new ChainedSearchRule<>(
                            new HistorySuggestRule<>(),
                            (input, request) ->
                                new SuggestRequestContextRequestProvider(
                                    input.suggestRequestContext(),
                                    request))),
                    (input, requests) ->
                        new SuggestRequestContextRequestsProvider(
                            input.suggestRequestContext(),
                            requests)),
                true));
        warmFields.add("history_suggest_timestamp");
        warmFields.add("history_suggest_request");

        rules.put(
            SuggestType.CONTENT,
            (input, callback) -> callback.completed(Collections.emptyList()));
        rules.put(
            SuggestType.FOLDERS,
            new TranslitRule<>(
                new ChainedSearchRule<>(
                    new ConcatResultsRule<>(
                        new ChainedSearchRule<>(
                            new FilesSuggestRule<>(SuggestType.FOLDERS, "dir"),
                            (input, request) ->
                                new SuggestRequestContextRequestProvider(
                                    input.suggestRequestContext(),
                                    request))),
                    (input, requests) ->
                        new SuggestRequestContextRequestsProvider(
                            input.suggestRequestContext(),
                            requests)),
                true));
        warmFields.add("aux_folder");
        warmFields.add("type");

        rules.put(
            SuggestType.FILES,
            new TranslitRule<>(
                new ChainedSearchRule<>(
                    new ConcatResultsRule<>(
                        new ChainedSearchRule<>(
                            new FilesSuggestRule<>(SuggestType.FILES, "file"),
                            (input, request) ->
                                new SuggestRequestContextRequestProvider(
                                    input.suggestRequestContext(),
                                    request))),
                    (input, requests) ->
                        new SuggestRequestContextRequestsProvider(
                            input.suggestRequestContext(),
                            requests)),
                true));
    }

    @Override
    public Set<String> warmUpKeyFields() {
        return warmFields;
    }

    @Override
    public Set<String> warmUpValueFields() {
        return warmFields;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        BasicSuggestRequestContext context =
            new BasicSuggestRequestContext(proxy, session, serviceName);
        Map<SuggestType, List<SuggestItem>> responses =
            new EnumMap<>(SuggestType.class);
        FutureCallback<Object> printer =
            new IdempotentFutureCallback<>(
                new SuggestResultCallback(
                    // TODO: URGENT: html escaping for text
                    new SuggestPrinter(context, HtmlHighlighter.INSTANCE),
                    responses));
        MultiFutureCallback<Object> callback =
            new MultiFutureCallback<>(printer);
        for (SuggestType type: context.types()) {
            rules.get(type).execute(
                context,
                new ResponseStoringCallback(
                    callback.newCallback(),
                    session.logger(),
                    type,
                    responses));
        }
        callback.done();
        Long timeoutBoxed = context.timeout();
        if (timeoutBoxed != null) {
            long now = System.currentTimeMillis();
            long timeElapsed = now - session.connection().requestStartTime();
            long timeout = timeoutBoxed.longValue() - timeElapsed;
            timeout = Math.max(timeout - TIMEOUT_OVERHEAD, MIN_TIMEOUT);
            timer.schedule(new TimeoutTask(printer), timeout);
        }
    }

    @Override
    public String warmupPostfix() {
        return "-suggest";
    }

    private static class ResponseStoringCallback
        extends ErrorSuppressingFutureCallback<List<SuggestItem>>
    {
        private final Logger logger;
        private final SuggestType type;
        private final Map<SuggestType, List<SuggestItem>> responses;

        // CSOFF: ParameterNumber
        ResponseStoringCallback(
            final FutureCallback<Object> callback,
            final Logger logger,
            final SuggestType type,
            final Map<SuggestType, List<SuggestItem>> responses)
        {
            super(callback, NULL_SUPPLIER);
            this.logger = logger;
            this.type = type;
            this.responses = responses;
        }
        // CSON: ParameterNumber

        @Override
        public void failed(final Exception e) {
            logger.log(
                Level.WARNING,
                "Failed to get results for " + type + SUGGEST,
                e);
            super.failed(e);
        }

        @Override
        public void completed(final List<SuggestItem> response) {
            if (response != null) {
                logger.info(
                    "Got " + response.size()
                    + " results for " + type + SUGGEST);
                synchronized (responses) {
                    responses.put(type, response);
                }
            }
            callback.completed(null);
        }
    }

    private static class SuggestResultCallback
        extends AbstractFilterFutureCallback<Object, SuggestResult>
    {
        private final Map<SuggestType, List<SuggestItem>> responses;

        SuggestResultCallback(
            final FutureCallback<? super SuggestResult> callback,
            final Map<SuggestType, List<SuggestItem>> responses)
        {
            super(callback);
            this.responses = responses;
        }

        @Override
        public void completed(final Object response) {
            SuggestResult result;
            synchronized (responses) {
                result = new SuggestResult(responses);
            }
            callback.completed(result);
        }
    }

    private static class TimeoutTask extends TimerTask {
        private final FutureCallback<?> callback;

        TimeoutTask(final FutureCallback<?> callback) {
            this.callback = callback;
        }

        @Override
        public void run() {
            callback.completed(null);
        }
    }
}

