package ru.yandex.search.disk.proxy.suggest;

import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonWriter;

public class SuggestPrinter
    extends AbstractProxySessionCallback<SuggestResult>
{
    private final SuggestRequestContext context;
    private final Highlighter highlighter;

    public SuggestPrinter(
        final SuggestRequestContext context,
        final Highlighter highlighter)
    {
        super(context.session());
        this.context = context;
        this.highlighter = highlighter;
    }

    @Override
    public void completed(final SuggestResult result) {
        Set<SuggestType> failedSuggests = EnumSet.noneOf(SuggestType.class);
        Set<String> ids = new HashSet<>();
        Map<SuggestType, List<SuggestItem>> suggestResults = result.suggests();
        List<List<SuggestItem>> deduplicated =
            new ArrayList<>(Math.min(suggestResults.size(), context.length()));
        int totalSize = 0;
        for (SuggestType type: context.types()) {
            List<SuggestItem> suggestItems = suggestResults.get(type);
            if (suggestItems == null) {
                failedSuggests.add(type);
            } else {
                // Do not add new groups to deduplicated map if there is no
                // space for this suggest group
                // TODO: put all items to single list, replacing less
                // scored with more scored items if they have same id,
                // then sort items according to their score
                if (deduplicated.size() < context.length()) {
                    List<SuggestItem> deduplicatedItems =
                        new ArrayList<>(suggestItems.size());
                    for (SuggestItem item: suggestItems) {
                        if (ids.add(item.id())) {
                            deduplicatedItems.add(item);
                        }
                    }
                    int size = deduplicatedItems.size();
                    if (size != 0) {
                        deduplicated.add(deduplicatedItems);
                        totalSize += size;
                    }
                }
            }
        }
        // Leave at least 1 suggest item for each group
        int pos = deduplicated.size() - 1;
        while (totalSize > context.length()) {
            List<SuggestItem> items = deduplicated.get(pos);
            int size = items.size();
            if (size > 1) {
                items.remove(size - 1);
                --totalSize;
            } else {
                --pos;
            }
        }
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = context.jsonType().create(sbw)) {
            writer.startObject();

            writer.key("retry-suggest-types");
            writer.startArray();
            for (SuggestType type: failedSuggests) {
                writer.value(type.toString());
            }
            writer.endArray();

            writer.key("suggest");
            writer.startArray();
            for (List<SuggestItem> items: deduplicated) {
                for (SuggestItem item: items) {
                    String text = item.text();
                    writer.startObject();
                    writer.key("type");
                    writer.value(item.type().toString());
                    writer.key("suggest-text");
                    writer.value(highlighter.highlight(item.request(), text));
                    writer.key("text");
                    writer.value(text);
                    for (Map.Entry<String, JsonObject> entry
                        : item.payload().entrySet())
                    {
                        writer.key(entry.getKey());
                        entry.getValue().writeValue(writer);
                    }
                    writer.endObject();
                }
            }
            writer.endArray();

            writer.endObject();
        } catch (IOException e) {
            failed(e);
            return;
        }
        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(context.session().acceptedCharset())));
    }
}

