package ru.yandex.search.disk.proxy.suggest.rules;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.disk.search.DiskParams;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.Proxy;
import ru.yandex.search.disk.proxy.suggest.BasicSuggestItem;
import ru.yandex.search.disk.proxy.suggest.SuggestItem;
import ru.yandex.search.disk.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.disk.proxy.suggest.SuggestType;
import ru.yandex.search.disk.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

public class FilesSuggestRule
    <T extends RequestProvider & SuggestRequestContextProvider>
    implements SearchRule<T, List<SuggestItem>>
{
    private static final String NAME = "name_tokenized";
    private static final FieldsTermsSupplierFactory NAME_FIELD =
        new FieldsTermsSupplierFactory(NAME);
    private static final FieldsTermsSupplierFactory FIELDS =
        new FieldsTermsSupplierFactory("folder_tokenized", NAME);
    private static final int MIN_LENGTH = 10;
    private static final String RESOURCE_ID = "resource_id";
    private static final String MEDIATYPE = "mediatype";

    private final SuggestType type;
    private final String fileType;

    public FilesSuggestRule(
        final SuggestType type,
        final String fileType)
    {
        this.type = type;
        this.fileType = fileType;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        SearchRequestText request = SearchRequestText.parseSuggest(
            input.request(),
            context.session().params().getLocale("locale", Locale.ROOT));
        if (request.isEmpty() || (type != SuggestType.FILES && !context.fileExtensions().isEmpty())) {
            callback.completed(Collections.emptyList());
        } else {
            execute(
                Math.max(MIN_LENGTH, context.length() << 1),
                new FileSuggestRequestContext(
                    context,
                    request,
                    type,
                    fileType,
                    callback));
        }
    }

    public static void execute(
        final int requestedLength,
        final FileSuggestRequestContext context)
        throws HttpException
    {
        StringBuilder sb = new StringBuilder(
            "aux_folder:(disk OR photounlim) AND type:");
        sb.append(context.fileType);
        if (!context.suggestRequestContext.fileExtensions().isEmpty()) {
            sb.append(" AND ext:(");
            sb.append(StringUtils.join(context.suggestRequestContext.fileExtensions(), ' '));
            sb.append(")");
        }

        SearchRequestText request = context.request;
        if (request.hasWords()) {
            sb.append(" AND (");
            // require at least one token present in file name
            request.fieldsQuery(sb, NAME_FIELD, ") OR (");
            sb.append(')');
            if (!request.singleWord()) {
                sb.append(" AND ");
                // tokens may also present in folder name
                request.fieldsQuery(sb, FIELDS);
            }
        }
        request.negationsQuery(sb, FIELDS);
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&sort=mtime&json-type=dollar&get="
                + "id,resource_id,key,name,mimetype,mediatype&skip-nulls");
        User user = context.suggestRequestContext.user();
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        query.append("text", new String(sb));
        if (context.suggestRequestContext.session().params().getBoolean(
            DiskParams.FAST_MOVED,
            false))
        {
            Proxy.addFastMovedDp(query);
        }

        query.append("length", requestedLength);
        context.suggestRequestContext.proxy().parallelRequest(
            context.suggestRequestContext.session(),
            context.suggestRequestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.suggestRequestContext.contextGenerator(),
            new Callback(context, requestedLength));
    }

    private static class FileSuggestRequestContext {
        private final SuggestRequestContext suggestRequestContext;
        private final SearchRequestText request;
        private final SuggestType type;
        private final String fileType;
        private final FutureCallback<? super List<SuggestItem>> callback;

        // CSOFF: ParameterNumber
        FileSuggestRequestContext(
            final SuggestRequestContext suggestRequestContext,
            final SearchRequestText request,
            final SuggestType type,
            final String fileType,
            final FutureCallback<? super List<SuggestItem>> callback)
        {
            this.suggestRequestContext = suggestRequestContext;
            this.request = request;
            this.type = type;
            this.fileType = fileType;
            this.callback = callback;
        }
        // CSON: ParameterNumber
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final FileSuggestRequestContext context;
        private final int requestedLength;

        Callback(
            final FileSuggestRequestContext context,
            final int requestedLength)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                context.suggestRequestContext.proxy().filterResources(
                    context.suggestRequestContext,
                    hits,
                    new FilteredResourcesCallback(
                        context,
                        requestedLength,
                        hits));
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private static class FilteredResourcesCallback
        extends AbstractFilterFutureCallback<
            Map<String, JsonMap>,
            List<SuggestItem>>
    {
        private final FileSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;

        FilteredResourcesCallback(
            final FileSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        @Override
        public void completed(final Map<String, JsonMap> resources) {
            try {
                String request = context.request.text();
                int count = 0;
                List<SuggestItem> items = new ArrayList<>(resources.size());
                Iterator<JsonObject> iter = hits.iterator();
                while (iter.hasNext()) {
                    JsonMap doc = iter.next().asMap();
                    String resourceId = doc.get(RESOURCE_ID).asStringOrNull();
                    if (resourceId != null) {
                        ++count;
                        JsonMap resource = resources.get(resourceId);
                        if (resource != null) {
                            JsonObject mediatype = resource.get("media_type");
                            if (mediatype == JsonNull.INSTANCE) {
                                doc.remove(MEDIATYPE);
                            } else {
                                doc.put(MEDIATYPE, mediatype);
                            }
                            items.add(
                                new BasicSuggestItem(
                                    doc.get("id").asString(),
                                    context.type,
                                    request,
                                    doc.remove("name").asString(),
                                    doc));
                        }
                    }
                }
                if (items.size() >= context.suggestRequestContext.length()
                    || count < requestedLength)
                {
                    callback.completed(items);
                } else {
                    execute(requestedLength << 1, context);
                }
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}

