package ru.yandex.search.disk.proxy.suggest.rules;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.disk.proxy.suggest.BasicSuggestItem;
import ru.yandex.search.disk.proxy.suggest.SuggestItem;
import ru.yandex.search.disk.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.disk.proxy.suggest.SuggestType;
import ru.yandex.search.disk.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class HistorySuggestRule
    <T extends RequestProvider & SuggestRequestContextProvider>
    implements SearchRule<T, List<SuggestItem>>
{
    private static final String HISTORY_SUGGEST_REQUEST =
        "history_suggest_request";
    private static final FieldsTermsSupplierFactory FIELD =
        new FieldsTermsSupplierFactory(HISTORY_SUGGEST_REQUEST);

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        SearchRequestText request = SearchRequestText.parseSuggest(
            input.request(),
            context.session().params().getLocale("locale", Locale.ROOT));
        StringBuilder sb = new StringBuilder();
        if (request.hasWords()) {
            // require at least one token present in file name
            request.fieldsQuery(sb, FIELD);
        } else {
            sb.append("has_history_suggest_request:1");
        }
        request.negationsQuery(sb, FIELD);
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&sort=history_suggest_timestamp"
                + "&json-type=dollar&get=id," + HISTORY_SUGGEST_REQUEST);
        query.append("prefix", context.user().prefix().toString());
        query.append("service", context.user().service());
        query.append("text", new String(sb));
        query.append("length", context.length());
        context.proxy().parallelRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(callback, request.text()));
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final String request;

        Callback(
            final FutureCallback<? super List<SuggestItem>> callback,
            final String request)
        {
            super(callback);
            this.request = request;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                List<SuggestItem> items = new ArrayList<>(hits.size());
                for (JsonObject hit: hits) {
                    JsonMap doc = hit.asMap();
                    items.add(
                        new BasicSuggestItem(
                            doc.get("id").asString(),
                            SuggestType.HISTORY,
                            request,
                            doc.get(HISTORY_SUGGEST_REQUEST).asString(),
                            JsonMap.EMPTY));
                }
                callback.completed(items);
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}

