package ru.yandex.search.district;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public enum DistrictFields {
    ID("id", false, DistrictEntityType.ALL),
    INDEX_TYPE("index_type", false, DistrictEntityType.ALL),
    CREATE_DATE("created_at", true, DistrictEntityType.ALL),
    ENTITY_ID("entity_id", DistrictEntityType.ALL),
    DISTRICT_ID("district_id", DistrictEntityType.ALL),
    USER_ID("user_id", true, DistrictEntityType.ALL),
    TEXT("text", true, DistrictEntityType.ALL),
    TEXT_WS("text_ws", true, DistrictEntityType.ALL),
    CITY_ID("city_id", true, DistrictEntityType.ALL),
    TAGS("tags", true, DistrictEntityType.ALL),
    ENTITY_TYPE("entity_type", DistrictEntityType.ALL),
    LIKES_COUNT("likes_cnt", DistrictEntityType.ALL),
    DISLIKES_COUNT("dislikes_cnt", DistrictEntityType.ALL),
    MENTION_USERS("mention_users", DistrictEntityType.ALL),
    // only posts
    EVENT_TYPE("event_type", true, DistrictEntityType.EVENT),
    TITLE("title", true, DistrictEntityType.EVENT),
    LOOKS_NICE("looks_nice", true, DistrictEntityType.EVENT),
    VIEWS_CNT("views_cnt", DistrictEntityType.EVENT),
    COMMENTS_COUNT("comments_cnt", DistrictEntityType.EVENT),
    LAST_COMMENT_DATE("last_comment_at", DistrictEntityType.EVENT),
    // only comments
    EVENT_ID("event_id", DistrictEntityType.COMMENT);

    public static final Set<DistrictFields> EVENT_FIELDS;
    public static final Set<DistrictFields> COMMENT_FIELDS;

    static {
        Set<DistrictFields> eventFields = new LinkedHashSet<>();
        Set<DistrictFields> commentFields = new LinkedHashSet<>();
        for (DistrictFields field: DistrictFields.values()) {
            if (field.entityiesTypes().contains(DistrictEntityType.ALL)
                || field.entityiesTypes().contains(DistrictEntityType.EVENT))
            {
                eventFields.add(field);
            }

            if (field.entityiesTypes().contains(DistrictEntityType.ALL)
                || field.entityiesTypes().contains(DistrictEntityType.COMMENT))
            {
                commentFields.add(field);
            }
        }

        EVENT_FIELDS = Collections.unmodifiableSet(eventFields);
        COMMENT_FIELDS = Collections.unmodifiableSet(commentFields);
    }

    private final String field;
    private final boolean preserve;
    private final boolean onlyPrefixed;
    private final String prefixedField;

    private final Set<DistrictEntityType> entityiesTypes;

    DistrictFields(
        final String field,
        final boolean hasPrefixedAlias,
        final DistrictEntityType... types)
    {
        this(field, true, "_p", types);
    }

    DistrictFields(
        final String field,
        final DistrictEntityType... types)
    {
        this(field, true, null, types);
    }

    // CSOFF: ParameterNumber
    DistrictFields(
        final String field,
        final boolean preserve,
        final String prefixedFieldPostfix,
        final DistrictEntityType... types)
    {
        this.field = field;
        this.preserve = preserve;
        this.entityiesTypes =
            Collections.unmodifiableSet(
                new LinkedHashSet<>(Arrays.asList(types)));
        // by default is prefixed
        if (prefixedFieldPostfix == null) {
            this.onlyPrefixed = true;
            this.prefixedField = field;
        } else {
            this.onlyPrefixed = false;
            this.prefixedField = field + prefixedFieldPostfix;
        }
    }
    // CSON: ParameterNumber

    public boolean onlyPrefixed() {
        return onlyPrefixed;
    }

    public String prefixedField() {
        return prefixedField;
    }

    public String field() {
        return field;
    }

    public boolean preserve() {
        return preserve;
    }

    public Set<DistrictEntityType> entityiesTypes() {
        return entityiesTypes;
    }

    public static Map<DistrictEntityType, List<String>> preserveFields() {
        Map<DistrictEntityType, List<String>> prsrvFields =
            new LinkedHashMap<>();

        Set<DistrictEntityType> types =
            new LinkedHashSet<>(Arrays.asList(DistrictEntityType.values()));
        types.remove(DistrictEntityType.ALL);
        for (DistrictFields df: DistrictFields.values()) {
            if (df.preserve()) {
                for (DistrictEntityType et: df.entityiesTypes()) {
                    if (et == DistrictEntityType.ALL) {
                        for (DistrictEntityType det: types) {
                            prsrvFields.computeIfAbsent(
                                det,
                                (k) -> new ArrayList<>()).add(df.field());
                        }
                    } else {
                        prsrvFields.computeIfAbsent(
                            et,
                            (k) -> new ArrayList<>()).add(df.field());
                    }
                }
            }
        }

        return Collections.unmodifiableMap(prsrvFields);
    }

    // CSOFF: ParameterNumber
    public static String cityBasedId(
        final DistrictEntityType type,
        final long prefix,
        final Long districtId,
        final String entityId)
    {
        //CSOFF: MagicNumber
        StringBuilder sb =
            new StringBuilder(
                entityId.length()
                    + type.entityType().length()
                    + 30);
        //CSON: MagicNumber
        sb.append("city_");
        sb.append(prefix);
        sb.append('_');
        sb.append(type.entityType());
        sb.append('_');
        if (districtId != null) {
            sb.append(districtId);
        } else {
            sb.append("all");
        }

        sb.append('_');
        sb.append(entityId);

        return sb.toString();
    }
    // CSON: ParameterNumber

    public static String id(
        final DistrictEntityType type,
        final long prefix,
        final String entityId)
    {
        //CSOFF: MagicNumber
        StringBuilder sb =
            new StringBuilder(
                entityId.length()
                    + type.entityType().length()
                    + 19);
        //CSON: MagicNumber

        sb.append(type.entityType());
        sb.append('_');
        sb.append(prefix);
        sb.append('_');
        sb.append(entityId);

        return sb.toString();
    }
}

