package ru.yandex.search.district.indexer;

import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.search.district.DistrictChangeType;
import ru.yandex.search.district.DistrictConstants;
import ru.yandex.search.district.DistrictEntityType;
import ru.yandex.search.district.DistrictFields;

public abstract class AbstractIndexContext {
    protected final int shard;
    protected final DistrictChangeType changeType;
    protected final DistrictEntityType entityType;
    protected final String entityId;
    protected final Map<String, String> data;
    protected final ProxySession session;
    protected final int value;
    protected final long lastCommentedDate;
    protected final long prefix;

    public AbstractIndexContext(
        final Long prefix,
        final ProxySession session,
        final JsonMap entry)
        throws JsonException
    {
        this.prefix = prefix;
        this.changeType =
            entry.getEnum(DistrictChangeType.class, "change_type");
        this.value = entry.getInt("value", 0);
        JsonMap fields = entry.getMap("fields");
        this.entityId = fields.getString(DistrictFields.ENTITY_ID.field());
        this.session = session;
        this.entityType = fields.getEnum(
            DistrictEntityType.class,
            DistrictFields.ENTITY_TYPE.field());
        this.lastCommentedDate =
            fields.getLong(DistrictFields.CREATE_DATE.field(), -1L);
        this.data = new LinkedHashMap<>(fields.size());
        Set<DistrictFields> fieldsSet;
        if (entityType == DistrictEntityType.EVENT) {
            fieldsSet = DistrictFields.EVENT_FIELDS;
        } else {
            fieldsSet = DistrictFields.COMMENT_FIELDS;
        }

        for (DistrictFields dField: fieldsSet) {
            if (fields.containsKey(dField.field())) {
                JsonObject obj = fields.get(dField.field());
                if (obj.type() == JsonObject.Type.LIST) {
                    StringBuilder sb = new StringBuilder();
                    for (JsonObject item: obj.asList()) {
                        sb.append(item.asString());
                        sb.append('\n');
                    }
                    if (sb.length() > 0) {
                        sb.setLength(sb.length() - 1);
                    }

                    data.put(dField.field(), sb.toString());
                } else if (obj.type() == JsonObject.Type.NULL) {
                    data.put(dField.field(), null);
                } else {
                    data.put(dField.field(), obj.asString());
                }
            }
        }

        if (entityType == DistrictEntityType.ALL) {
            throw new JsonException("Invalid entity type ALL");
        }

        this.shard = (int) (prefix % DistrictConstants.MAX_SHARDS);
    }

    protected abstract String service();

    public abstract String id();

    public abstract String lucenePostId(final String eventId);

    public long prefix() {
        return prefix;
    }

    public DistrictChangeType changeType() {
        return changeType;
    }

    public DistrictEntityType entityType() {
        return entityType;
    }

    public Map<String, String> data() {
        return data;
    }

    public ProxySession session() {
        return session;
    }

    public int shard() {
        return shard;
    }

    public String entityId() {
        return entityId;
    }

    public int value() {
        return value;
    }

    public long lastCommentedDate() {
        return lastCommentedDate;
    }

    public StringBuilder uri(final StringBuilder uri) {
        uri.append("&service=");
        uri.append(service());
        uri.append("&prefix=");
        uri.append(prefix());
        uri.append("&producer-session=");
        uri.append(
            session.context().getAttribute(HttpServer.SESSION_ID));
        uri.append("&transfer-timestamp=");
        uri.append(System.currentTimeMillis());
        uri.append("&entity-type=");
        uri.append(entityType);
        uri.append("&change-type=");
        uri.append(changeType.toString().toLowerCase(Locale.ROOT));
        uri.append("&url=");
        uri.append(id());

        return uri;
    }
}
